package com.yifu.cloud.plus.v1.business.controller.cert;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.business.dto.vacation.UpdateVacationRedundancyDTO;
import com.yifu.cloud.plus.v1.business.entity.cert.TCertInfo;
import com.yifu.cloud.plus.v1.business.service.cert.TCertInfoService;
import com.yifu.cloud.plus.v1.business.vo.cert.TCertInfoVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.List;


/**
 * 人员证书信息表
 *
 * @author fxj
 * @date 2021-08-16 15:34:16
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tcertinfo")
@Tag(name = "人员证书信息表")
public class TCertInfoController {

	private final TCertInfoService tCertInfoService;

	/**
	 * 简单分页查询
	 *
	 * @param page      分页对象
	 * @param tCertInfo 人员证书信息表
	 * @return
	 */
	@Operation(summary = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TCertInfo>> getTCertInfoPage(Page<TCertInfo> page, TCertInfo tCertInfo) {
		if (StringUtils.isNotBlank(tCertInfo.getLabel())) {
			String empTag = String.format(CommonConstants.TAG_FROMAT, Integer.valueOf(tCertInfo.getLabel()));
			tCertInfo.setLabel(empTag);
		}
		return new R<>(tCertInfoService.getTCertInfoPage(page, tCertInfo));
	}

	/**
	 * 简单分页查询
	 *
	 * @param tCertInfo 人员证书信息表
	 * @return
	 */
	@Operation(summary = "导出查询")
	@GetMapping("/getCertInfoForExport")
	public R<List<TCertInfo>> getCertInfoForExport(TCertInfo tCertInfo) {
		if (StringUtils.isNotBlank(tCertInfo.getLabel())) {
			String empTag = String.format(CommonConstants.TAG_FROMAT, Integer.valueOf(tCertInfo.getLabel()));
			tCertInfo.setLabel(empTag);
		}
		return R.ok(tCertInfoService.getCertInfoForExport(tCertInfo));
	}

	/**
	 * 通过id查询单条记录
	 *
	 * @param id
	 * @return R
	 */
	@Operation(summary = "id查询")
	@GetMapping("/{id}")
	public R<TCertInfoVo> getById(@PathVariable("id") String id) {
		return tCertInfoService.getVoById(id);
	}

	/**
	 * 通过身份证号查询单条记录
	 *
	 * @param empIdcard
	 * @return R
	 */
	@Operation(summary = "empIdcard查询所有证件信息")
	@GetMapping("/getByempIdcard")
	public R<List<TCertInfoVo>> getByempIdcard(@RequestParam(name = "empIdcard") String empIdcard) {
		return tCertInfoService.getByEmpIdcard(empIdcard);
	}

	/**
	 * 新增记录
	 *
	 * @param tCertInfo
	 * @return R
	 */
	@Operation(summary = "新增(yifu-hro-business:tcertinfo_add)")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('yifu-hro-business:tcertinfo_add')")
	public R<TCertInfo> save(@Valid @RequestBody TCertInfo tCertInfo) {
		YifuUser user = SecurityUtils.getUser();
		if (null == user) {
			return R.failed("获取用户信息失败！");
		}
		tCertInfo.setStatus(CommonConstants.ZERO_STRING);
		tCertInfo.setCreateUser(user.getNickname());
		tCertInfo.setCreateTime(LocalDateTime.now());
		return tCertInfoService.saveAsso(tCertInfo);
	}

	/**
	 * 修改记录
	 *
	 * @param tCertInfo
	 * @return R
	 */
	@Operation(summary = "修改(yifu-hro-business:tcertinfo_edit)")
	@SysLog("修改人员证书信息表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('yifu-hro-business:tcertinfo_edit')")
	public R<String> update(@RequestBody TCertInfo tCertInfo) {
		return tCertInfoService.updateByIdDiy(tCertInfo);
	}

	/**
	 * 通过id删除一条记录
	 *
	 * @param id
	 * @return R
	 */
	@Operation(summary = "删除(yifu-hro-business:tcertinfo_del)")
	@SysLog("删除人员证书信息表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('yifu-hro-business:tcertinfo_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return tCertInfoService.removeByIdDiy(id);
	}

	/**
	 * b端人员证件信息批量导入
	 *
	 * @param jsonStr
	 * @description: b端人员证件信息批量导入
	 * @author: fxj
	 * @date: 2021/8/10
	 */
	@Operation(summary = "批量导入：b端人员证件信息批量导入")
	@PutMapping("/batchImportCertByJsonStr")
	public R<List<ErrorMessage>> batchImportCertByJsonStr(@RequestBody String jsonStr) {
		return tCertInfoService.batchImportCertByJsonStr(jsonStr);
	}

	/**
	 * b端人员证件信息批量更新
	 *
	 * @param jsonStr
	 * @description: b端人员证件信息批量更新
	 * @author: fxj
	 * @date: 2021/8/10
	 */
	@Operation(summary = "批量更新：b端人员证件信息批量更新")
	@PutMapping("/batchUpdateCertByJsonStr")
	public R<List<ErrorMessage>> batchUpdateCertByJsonStr(@RequestBody String jsonStr) {
		return tCertInfoService.batchUpdateCertByJsonStr(jsonStr);
	}


	/**
	 * 定时任务更新证件状态信息
	 *
	 * @return
	 * @date 2021-08-23
	 * @author: fxj
	 */
	@Inner
	@Operation(summary = "B端定时更新证件状态")
	@PostMapping("/inner/updateCertStatusTask")
	public R<Boolean> updateCertStatusTask() {
		return tCertInfoService.updateCertStatusTask();
	}

	/**
	 * 更新证件信息中的冗余信息
	 *
	 * @param dtoList
	 * @return
	 */
	@Inner
	@PostMapping("/inner/updateRedundancyEmpOfCert")
	public void updateRedundancyEmpOfCert(@RequestBody List<UpdateVacationRedundancyDTO> dtoList) {
		tCertInfoService.updateRedundancyEmpOfCert(dtoList);
	}

}
