/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.check.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.check.entity.TCheckBankNo;
import com.yifu.cloud.plus.v1.check.service.TCheckBankNoService;
import com.yifu.cloud.plus.v1.check.vo.CheckBankNoVo;
import com.yifu.cloud.plus.v1.check.vo.CheckBatchVo;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;
import java.util.concurrent.ExecutionException;


/**
 * 银行卡卡号 校验
 *
 * @author fxj
 * @date 2022-06-20 19:56:33
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tcheckbankno" )
@Tag(name = "银行卡卡号 校验管理")
public class TCheckBankNoController {

    private final TCheckBankNoService tCheckBankNoService;

    /**
     * 分页查询
     * @param page 分页对象
     * @param tCheckBankNo 银行卡卡号 校验
     * @return
     */
    @Operation(summary = "分页查询", description = "分页查询")
    @GetMapping("/page" )
    public R getTCheckBankNoPage(Page page, TCheckBankNo tCheckBankNo) {
        return R.ok(tCheckBankNoService.page(page, Wrappers.query(tCheckBankNo)));
    }


    /**
     * 通过id查询银行卡卡号 校验
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询")
    @GetMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_tcheckbankno_get')" )
    public R getById(@PathVariable("id" ) String id) {
        return R.ok(tCheckBankNoService.getById(id));
    }

    /**
     * 新增银行卡卡号 校验
     * @param tCheckBankNo 银行卡卡号 校验
     * @return R
     */
    @Operation(summary = "新增银行卡卡号 校验", description = "新增银行卡卡号 校验")
    @SysLog("新增银行卡卡号 校验" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('demo_tcheckbankno_add')" )
    public R save(@RequestBody TCheckBankNo tCheckBankNo) {
        return R.ok(tCheckBankNoService.save(tCheckBankNo));
    }

    /**
     * 修改银行卡卡号 校验
     * @param tCheckBankNo 银行卡卡号 校验
     * @return R
     */
    @Operation(summary = "修改银行卡卡号 校验", description = "修改银行卡卡号 校验")
    @SysLog("修改银行卡卡号 校验" )
    @PutMapping
    @PreAuthorize("@pms.hasPermission('demo_tcheckbankno_edit')" )
    public R updateById(@RequestBody TCheckBankNo tCheckBankNo) {
        return R.ok(tCheckBankNoService.updateById(tCheckBankNo));
    }

    /**
     * 通过id删除银行卡卡号 校验
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除银行卡卡号 校验", description = "通过id删除银行卡卡号 校验")
    @SysLog("通过id删除银行卡卡号 校验" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_tcheckbankno_del')" )
    public R removeById(@PathVariable String id) {
        return R.ok(tCheckBankNoService.removeById(id));
    }

	/**
	 * 银行卡号校验
	 * @param  checkBankNo	姓名	是	[string] 身份证号码，限单个	是	[string] 银行卡号，限单个	是	[string]银行预留手机号，仅支持国内11位号码
	 */
	@Operation(description = "银行卡号校验")
	@SysLog("银行卡号校验" )
	@Inner
	@PostMapping("/inner/checkBankNo" )
	public CheckBankNoVo checkBankNoInner(@RequestBody TCheckBankNo checkBankNo) {
		return tCheckBankNoService.checkBankNoTwo(checkBankNo.getName(),checkBankNo.getBankNo());
	}

	/**
	 * @param empName 姓名
	 * @param bankNo 银行卡号
	 * @param deptId 项目id
	 * @Description:  C端预入职银行卡号校验
	 * @Author: hgw
	 * @Date: 2024/7/16 14:58
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(description = "C端预入职银行卡号校验")
	@SysLog("C端预入职银行卡号校验" )
	@GetMapping("/checkPreBankNo" )
	public R<String> checkPreBankNo(@RequestParam String empName,@RequestParam String bankNo, @RequestParam String deptId) {
		return tCheckBankNoService.checkPreBankNo(empName, bankNo, deptId);
	}

	/**
	 * @Description: 批量银行卡号校验
	 * @Author: hgw
	 * @Date: 2023/2/9 9:57
	 * @return: com.yifu.cloud.plus.v1.check.vo.CheckBankNoVo
	 **/
	@Operation(description = "批量银行卡号校验")
	@SysLog("批量银行卡号校验" )
	@Inner
	@PostMapping("/inner/checkBankNoBatch" )
	public CheckBatchVo checkBankNoBatch(@RequestBody List<TCheckBankNo> list) throws ExecutionException, InterruptedException{
		return tCheckBankNoService.checkBankNoBatch(list);
	}

	/**
	 * @Description: 批量银行卡号校验返回含名字
	 * @Author: hgw
	 * @Date: 2024-10-23 18:13:15
	 * @return: com.yifu.cloud.plus.v1.check.vo.CheckBankNoVo
	 **/
	@Operation(description = "批量银行卡号校验返回含名字")
	@SysLog("批量银行卡号校验返回含名字" )
	@Inner
	@PostMapping("/inner/checkBankNoAndNameBatch" )
	public CheckBatchVo checkBankNoAndNameBatch(@RequestBody List<TCheckBankNo> list) throws ExecutionException, InterruptedException{
		return tCheckBankNoService.checkBankNoAndNameBatch(list);
	}

	/**
	 * @Description: 代发户调用银行卡校验--如果其他接口也要调用，加上信息
	 * @Author: hgw
	 * @Date: 2023/1/13 9:41
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.check.entity.TCheckBankNo>
	 **/
	@Operation(description = "调用银行卡校验")
	@SysLog("调用银行卡校验" )
	@PostMapping("/checkBankNo" )
	public R<TCheckBankNo> checkBankNo(@RequestParam String name, @RequestParam String bankNo) {
		return tCheckBankNoService.checkBankNo(name, bankNo);
	}

	/**
	 * 银行卡号校验
	 * @param  checkBankNo	姓名	是	[string] 身份证号码，限单个	是	[string] 银行卡号，限单个	是	[string]银行预留手机号，仅支持国内11位号码
	 */
	@Operation(description = "银行卡号校验")
	@SysLog("银行卡号校验" )
	@Inner
	@PostMapping("/inner/checkBankNoThree" )
	public CheckBankNoVo checkBankNoThreeInner(@RequestBody TCheckBankNo checkBankNo) {
		return tCheckBankNoService.checkBankNoThree(checkBankNo.getName(),checkBankNo.getBankNo(),checkBankNo.getIdNum());
	}

	/**
	 * 批量新增员工工作信息
	 *
	 * @param file
	 * @return
	 * @Author fxj
	 * @Date 2019-08-16
	 **/
	@SneakyThrows
	@Operation(description = "批量新增员工工作信息 hasPermission('archives_tempworkrecording_importListAdd')")
	@SysLog("批量新增员工工作信息")
	@PostMapping("/importListAdd")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file) {
		return tCheckBankNoService.importDiy(file.getInputStream());
	}
}
