/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.check.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.check.entity.TCheckMobile;
import com.yifu.cloud.plus.v1.check.service.TCheckMobileService;
import com.yifu.cloud.plus.v1.check.vo.CheckBatchVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;


/**
 * 手机号码校验
 *
 * @author fxj
 * @date 2022-06-20 17:29:03
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tcheckmobile" )
@Tag(name = "手机号码校验管理")
public class TCheckMobileController {

    private final TCheckMobileService tCheckMobileService;

    /**
     * 分页查询
     * @param page 分页对象
     * @param tCheckMobile 手机号码校验
     * @return
     */
    @Operation(summary = "分页查询", description = "分页查询")
    @GetMapping("/page" )
    public R getTCheckMobilePage(Page page, TCheckMobile tCheckMobile) {
        return R.ok(tCheckMobileService.page(page, Wrappers.query(tCheckMobile)));
    }


    /**
     * 通过id查询手机号码校验
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询")
    @GetMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_tcheckmobile_get')" )
    public R getById(@PathVariable("id" ) String id) {
        return R.ok(tCheckMobileService.getById(id));
    }

    /**
     * 新增手机号码校验
     * @param tCheckMobile 手机号码校验
     * @return R
     */
    @Operation(summary = "新增手机号码校验", description = "新增手机号码校验")
    @SysLog("新增手机号码校验" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('demo_tcheckmobile_add')" )
    public R save(@RequestBody TCheckMobile tCheckMobile) {
        return R.ok(tCheckMobileService.save(tCheckMobile));
    }

    /**
     * 修改手机号码校验
     * @param tCheckMobile 手机号码校验
     * @return R
     */
    @Operation(summary = "修改手机号码校验", description = "修改手机号码校验")
    @SysLog("修改手机号码校验" )
    @PutMapping
    @PreAuthorize("@pms.hasPermission('demo_tcheckmobile_edit')" )
    public R updateById(@RequestBody TCheckMobile tCheckMobile) {
        return R.ok(tCheckMobileService.updateById(tCheckMobile));
    }

    /**
     * 通过id删除手机号码校验
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除手机号码校验", description = "通过id删除手机号码校验")
    @SysLog("通过id删除手机号码校验" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_tcheckmobile_del')" )
    public R removeById(@PathVariable String id) {
        return R.ok(tCheckMobileService.removeById(id));
    }

	/**
	 * 手机号码状态校验
	 * @param mobiles  检测手机号，多个手机号码用英文半角逗号隔开，仅支持国内号码
	 * @return R
	 */
	@Operation(description = "检测手机号，多个手机号码用英文半角逗号隔开，仅支持国内号码")
	@SysLog("检测手机号状态" )
	@GetMapping("/checkMobiles" )
	public R<Map<String,TCheckMobile>> checkMobiles(String mobiles) {
		return tCheckMobileService.checkMobiles(mobiles);
	}

	/**
	 * 手机号码状态校验
	 * @param mobiles  检测手机号，多个手机号码用英文半角逗号隔开，仅支持国内号码
	 * @return R
	 */
	@Operation(description = "检测手机号，多个手机号码用英文半角逗号隔开，仅支持国内号码")
	@SysLog("检测手机号状态" )
	@Inner
	@PostMapping("/inner/checkMobiles" )
	public R<Map<String,TCheckMobile>> checkMobilesInner(@RequestParam String mobiles) {
		return tCheckMobileService.checkMobiles(mobiles);
	}

	/**
	 * @Description: 单个检测手机号
	 * @Author: hgw
	 * @Date: 2022/6/29 15:08
	 * @return: boolean
	 **/
	@Operation(description = "单个检测手机号，仅支持国内号码")
	@SysLog("单个检测手机号状态" )
	@Inner
	@PostMapping("/inner/checkOneMobile" )
	public TCheckMobile checkOneMobile(@RequestBody TCheckMobile checkMobile) {
		if (Common.isNotNull(checkMobile.getMobile())) {
			return tCheckMobileService.checkOneMobile(checkMobile.getMobile());
		} else {
			checkMobile.setStatus(CommonConstants.ZERO_STRING);
			checkMobile.setMessage("手机号必填");
			return checkMobile;
		}
	}

	/**
	 * @Description: 批量手机号校验
	 * @Author: hgw
	 * @Date: 2023-2-9 11:23:10
	 * @return: com.yifu.cloud.plus.v1.check.vo.CheckBankNoVo
	 **/
	@Operation(description = "批量手机号校验")
	@SysLog("批量手机号校验" )
	@Inner
	@PostMapping("/inner/checkMobileBatch" )
	public CheckBatchVo checkMobileBatch(@RequestBody List<String> list) {
		return tCheckMobileService.checkMobileBatch(list);
	}
	/**
	 * @Author fxj
	 * @Description 批量手机号校验
	 * @Date 18:20 2023/3/1
	**/
	@Operation(description = "批量手机号校验")
	@SysLog("批量手机号校验" )
	@PostMapping("/checkMobileBatch" )
	public CheckBatchVo checkMobileBatchs(@RequestBody List<String> list) {
		return tCheckMobileService.checkMobileBatch(list);
	}

	/**
	 * @Description: C端预入职校验手机号
	 * @Author: hgw
	 * @Date: 2024-6-27 15:49:32
	 * @return: com.yifu.cloud.v1.common.core.util.R
	 **/
	@Operation(description = "C端预入职校验手机号")
	@SysLog("C端预入职校验手机号")
	@GetMapping("/checkPrePhone")
	public R<String> checkPrePhone(@RequestParam String phone, @RequestParam String deptId) {
		return tCheckMobileService.checkPrePhone(phone, deptId);
	}

}
