/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CacheConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.ResultConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.redis.RedisDistributedLock;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.social.entity.TAutoPaymentDetail;
import com.yifu.cloud.plus.v1.yifu.social.entity.TDispatchInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.TAutoPaymentDetailService;
import com.yifu.cloud.plus.v1.yifu.social.vo.TAutoPaymentDetailSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 自动化实缴明细表
 *
 * @author fxj
 * @date 2024-05-24 10:56:42
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tautopaymentdetail")
@Tag(name = "自动化实缴明细表")
public class TAutoPaymentDetailController {

	private final TAutoPaymentDetailService tAutoPaymentDetailService;


	/**
	 * 简单分页查询
	 *
	 * @param page               分页对象
	 * @param tAutoPaymentDetail 自动化实缴记录表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TAutoPaymentDetail>> getTAutoPaymentDetailPage(Page<TAutoPaymentDetail> page, TAutoPaymentDetailSearchVo tAutoPaymentDetail) {
		return new R<>(tAutoPaymentDetailService.getTAutoPaymentDetailPage(page, tAutoPaymentDetail));
	}

	/**
	 * 不分页查询
	 *
	 * @param tAutoPaymentDetail 自动化实缴记录表
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	public R<List<TAutoPaymentDetail>> getTAutoPaymentDetailNoPage(@RequestBody TAutoPaymentDetailSearchVo tAutoPaymentDetail) {
		return R.ok(tAutoPaymentDetailService.noPageDiy(tAutoPaymentDetail));
	}

	/**
	 * 通过id查询自动化实缴记录表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('social_tautopaymentdetail_get')")
	@GetMapping("/{id}")
	public R<TAutoPaymentDetail> getById(@PathVariable("id") String id) {
		return R.ok(tAutoPaymentDetailService.getById(id));
	}

	/**
	 * 新增自动化实缴记录表
	 *
	 * @param tAutoPaymentDetail 自动化实缴记录表
	 * @return R
	 */
	@Operation(summary = "新增自动化实缴记录表", description = "新增自动化实缴记录表：hasPermission('social_tautopaymentdetail_add')")
	@SysLog("新增自动化实缴记录表")
	@PostMapping
	public R<Boolean> save(@RequestBody TAutoPaymentDetail tAutoPaymentDetail) {
		return R.ok(tAutoPaymentDetailService.save(tAutoPaymentDetail));
	}

	/**
	 * 修改自动化实缴记录表
	 *
	 * @param tAutoPaymentDetail 自动化实缴记录表
	 * @return R
	 */
	@Operation(summary = "修改自动化实缴记录表", description = "修改自动化实缴记录表：hasPermission('social_tautopaymentdetail_edit')")
	@SysLog("修改自动化实缴记录表")
	@PutMapping
	public R<Boolean> updateById(@RequestBody TAutoPaymentDetail tAutoPaymentDetail) {
		return R.ok(tAutoPaymentDetailService.updateById(tAutoPaymentDetail));
	}

	/**
	 * 通过id删除自动化实缴记录表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除自动化实缴记录表", description = "通过id删除自动化实缴记录表：hasPermission('social_tautopaymentdetail_del')")
	@SysLog("通过id删除自动化实缴记录表")
	@DeleteMapping("/{id}")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tAutoPaymentDetailService.removeById(id));
	}

	/**
	 * 自动化实缴记录表 批量导入
	 *
	 * @author fxj
	 * @date 2024-05-24 10:56:42
	 **/
	@SneakyThrows
	@Operation(description = "标记重新办理数据")
	@SysLog("标记重新办理数据")
	@PostMapping("/importListAdd")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file, @RequestParam String parentId) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)) {
			return R.failed(CommonConstants.USER_FAIL);
		}
		// 获取redis分布式事务锁
		String key = CacheConstants.FUND_IMPORT_HANDLE_LOCK + CommonConstants.DOWN_LINE_STRING + user.getId();
		String requestId;
		try {
			requestId = RedisDistributedLock.getLock(key, "10");
		} catch (Exception e) {
			throw new RuntimeException(ResultConstants.NO_GETLOCK_DATA + CommonConstants.DOWN_LINE_STRING + e.getMessage());
		}
		try {
			if (Common.isNotNull(requestId)) {
				return tAutoPaymentDetailService.importDiy(file.getInputStream(),parentId);
			} else {
				return R.failed(ResultConstants.NO_GETLOCK_DATA);
			}
		} finally {
			//主动释放锁
			RedisDistributedLock.unlock(key, requestId);
		}

	}

	/**
	 * 自动化实缴记录表 批量导出
	 *
	 * @author fxj
	 * @date 2024-05-24 10:56:42
	 **/
	@Operation(description = "导出自动化实缴记录表 hasPermission('social_tautopaymentdetail-export')")
	@PostMapping("/export")
	public void export(HttpServletResponse response, @RequestBody TAutoPaymentDetailSearchVo searchVo) {
		tAutoPaymentDetailService.listExport(response, searchVo);
	}
}
