/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.social.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ArrayUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.read.listener.ReadListener;
import com.alibaba.excel.read.metadata.holder.ReadRowHolder;
import com.alibaba.excel.util.ListUtils;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.*;
import com.yifu.cloud.plus.v1.yifu.common.mybatis.base.BaseEntity;
import com.yifu.cloud.plus.v1.yifu.social.entity.THolidayInfo;
import com.yifu.cloud.plus.v1.yifu.social.mapper.THolidayInfoMapper;
import com.yifu.cloud.plus.v1.yifu.social.service.THolidayInfoService;
import com.yifu.cloud.plus.v1.yifu.social.vo.TEmployeeInsuranceWorkDayVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.THolidayInfoSearchVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.THolidayInfoVo;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.*;

/**
 * @author fxj
 * @date 2024-05-08 11:52:08
 */
@Log4j2
@Service
public class THolidayInfoServiceImpl extends ServiceImpl<THolidayInfoMapper, THolidayInfo> implements THolidayInfoService {
	/**
	 * 简单分页查询
	 *
	 * @param tHolidayInfo
	 * @return
	 */
	@Override
	public IPage<THolidayInfo> getTHolidayInfoPage(Page<THolidayInfo> page, THolidayInfoSearchVo tHolidayInfo) {
		return baseMapper.getTHolidayInfoPage(page, tHolidayInfo);
	}

	/**
	 * 批量导出
	 *
	 * @param searchVo
	 * @return
	 */
	@Override
	public void listExport(HttpServletResponse response, THolidayInfoSearchVo searchVo) {
		String fileName = "批量导出" + DateUtil.getThisTime() + ".xlsx";
		//获取要导出的列表
		List<THolidayInfo> list = new ArrayList<>();
		long count = noPageCountDiy(searchVo);
		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName, CommonConstants.UTF8));
			// 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
			//EasyExcel.write(out, TEmpBadRecord.class).sheet("不良记录").doWrite(list);
			ExcelWriter excelWriter = EasyExcel.write(out, THolidayInfo.class).build();
			int index = 0;
			if (count > CommonConstants.ZERO_INT) {
				for (int i = 0; i <= count; ) {
					// 获取实际记录
					searchVo.setLimitStart(i);
					searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
					list = noPageDiy(searchVo);
					if (Common.isNotNull(list)) {
						ExcelUtil<THolidayInfo> util = new ExcelUtil<>(THolidayInfo.class);
						for (THolidayInfo vo : list) {
							util.convertEntity(vo, null, null, null);
						}
					}
					if (Common.isNotNull(list)) {
						WriteSheet writeSheet = EasyExcel.writerSheet("" + index).build();
						excelWriter.write(list, writeSheet);
						index++;
					}
					i = i + CommonConstants.EXCEL_EXPORT_LIMIT;
					if (Common.isNotNull(list)) {
						list.clear();
					}
				}
			} else {
				WriteSheet writeSheet = EasyExcel.writerSheet("" + index).build();
				excelWriter.write(list, writeSheet);
			}
			if (Common.isNotNull(list)) {
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		} catch (Exception e) {
			log.error("执行异常", e);
		} finally {
			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	@Override
	public List<THolidayInfo> noPageDiy(THolidayInfoSearchVo searchVo) {
		LambdaQueryWrapper<THolidayInfo> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(THolidayInfo::getId, idList);
		}
		if (searchVo.getLimitStart() >= 0 && searchVo.getLimitEnd() > 0) {
			wrapper.last(" limit " + searchVo.getLimitStart() + "," + searchVo.getLimitEnd());
		}
		wrapper.orderByDesc(BaseEntity::getCreateTime);
		return baseMapper.selectList(wrapper);
	}

	private Long noPageCountDiy(THolidayInfoSearchVo searchVo) {
		LambdaQueryWrapper<THolidayInfo> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(THolidayInfo::getId, idList);
		}
		return baseMapper.selectCount(wrapper);
	}

	private LambdaQueryWrapper buildQueryWrapper(THolidayInfoSearchVo entity) {
		LambdaQueryWrapper<THolidayInfo> wrapper = Wrappers.lambdaQuery();
		if (ArrayUtil.isNotEmpty(entity.getCreateTimes())) {
			wrapper.ge(THolidayInfo::getCreateTime, entity.getCreateTimes()[0])
					.le(THolidayInfo::getCreateTime,
							entity.getCreateTimes()[1]);
		}
		if (Common.isNotNull(entity.getCreateName())) {
			wrapper.eq(THolidayInfo::getCreateName, entity.getCreateName());
		}
		return wrapper;
	}

	@Override
	public R<List<ErrorMessage>> importDiy(InputStream inputStream) {
		List<ErrorMessage> errorMessageList = new ArrayList<>();
		ExcelUtil<THolidayInfoVo> util1 = new ExcelUtil<>(THolidayInfoVo.class);
		;
		// 写法2：
		// 匿名内部类 不用额外写一个DemoDataListener
		// 这里 需要指定读用哪个class去读，然后读取第一个sheet 文件流会自动关闭
		try {
			EasyExcel.read(inputStream, THolidayInfoVo.class, new ReadListener<THolidayInfoVo>() {
				/**
				 * 单次缓存的数据量
				 */
				public static final int BATCH_COUNT = CommonConstants.BATCH_COUNT;
				/**
				 *临时存储
				 */
				private List<THolidayInfoVo> cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);

				@Override
				public void invoke(THolidayInfoVo data, AnalysisContext context) {
					ReadRowHolder readRowHolder = context.readRowHolder();
					Integer rowIndex = readRowHolder.getRowIndex();
					data.setRowIndex(rowIndex + 1);
					ErrorMessage errorMessage = util1.checkEntity(data, data.getRowIndex());
					if (Common.isNotNull(errorMessage)) {
						errorMessageList.add(errorMessage);
					} else {
						cachedDataList.add(data);
					}
					if (cachedDataList.size() >= BATCH_COUNT) {
						saveData();
						// 存储完成清理 list
						cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);
					}
				}

				@Override
				public void doAfterAllAnalysed(AnalysisContext context) {
					saveData();
				}

				/**
				 * 加上存储数据库
				 */
				private void saveData() {
					log.info("{}条数据，开始存储数据库！", cachedDataList.size());
					importTHolidayInfo(cachedDataList, errorMessageList);
					log.info("存储数据库成功！");
				}
			}).sheet().doRead();
		} catch (Exception e) {
			log.error(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR, e);
			return R.failed(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR);
		}
		return R.ok(errorMessageList);
	}

	private void importTHolidayInfo(List<THolidayInfoVo> excelVOList, List<ErrorMessage> errorMessageList) {
		// 个性化校验逻辑
		ErrorMessage errorMsg;
		// 执行数据插入操作 组装
		for (int i = 0; i < excelVOList.size(); i++) {
			THolidayInfoVo excel = excelVOList.get(i);
			// 数据合法情况 TODO

			// 插入
			insertExcel(excel);
			errorMessageList.add(new ErrorMessage(excel.getRowIndex(), CommonConstants.SAVE_SUCCESS));
		}
	}

	/**
	 * 插入excel bad record
	 */
	private void insertExcel(THolidayInfoVo excel) {
		THolidayInfo insert = new THolidayInfo();
		BeanUtil.copyProperties(excel, insert);
		this.save(insert);
	}

	@Override
	public void createHolidayInfo() {
		baseMapper.delete(Wrappers.<THolidayInfo>query().lambda());
		List<Map<String,Object>>  json = HolidayCrawler.initHolidays();
		List<THolidayInfo> saveList = new ArrayList<>();
		THolidayInfo save;
		for(Map<String,Object> d:json){
			save = new THolidayInfo();
			try {
				save.setDate(DateUtil.stringToDate((String) d.get("date").toString(),DateUtil.ISO_DATE_FORMAT));
			}catch (Exception e){
				log.error("假期表初始化异常："+d.get("date"),e);
				continue;
			}
			save.setDay(DateUtil.getDay(save.getDate()));
			save.setMonth(DateUtil.getMonth(save.getDate()));
			save.setYear(DateUtil.getYear(save.getDate()));
			save.setCreateBy("1");
			save.setCreateName("admin");
			save.setCreateTime(LocalDateTime.now());
			save.setUpdateBy("1");
			saveList.add(save);
		}
		if (Common.isNotNull(saveList)){
			this.saveBatch(saveList);
		}
		log.info(DateUtil.getYear(new Date())+"年假期表初始化完成：");
	}

	@Override
	public TEmployeeInsuranceWorkDayVo selectBeforeOrAfterWorkDay(TEmployeeInsuranceWorkDayVo vo) {
		//获取当年所有假期配置表
		Date date = new Date();
		List<THolidayInfo> holidayInfos = baseMapper.selectList(Wrappers.<THolidayInfo>query().lambda()
				.eq(THolidayInfo::getYear,DateUtil.getYear(date)));
		Map<Date,String> holidayMap = new HashMap<>();
		if (Common.isNotNull(holidayInfos)){
			for (THolidayInfo h:holidayInfos){
				holidayMap.put(h.getDate(),h.getDay());
			}
		}
		getWorkDay(holidayMap,vo);
		return vo;
	}

	@Override
	public TEmployeeInsuranceWorkDayVo getInitBeforeOrAfterWorkDay(TEmployeeInsuranceWorkDayVo vo) {
		//获取当年所有假期配置表
		Date date = new Date();
		List<THolidayInfo> holidayInfos = baseMapper.selectList(Wrappers.<THolidayInfo>query().lambda()
				.eq(THolidayInfo::getYear,DateUtil.getYear(date)));
		Map<Date,String> holidayMap = new HashMap<>();
		if (Common.isNotNull(holidayInfos)){
			for (THolidayInfo h:holidayInfos){
				holidayMap.put(h.getDate(),h.getDay());
			}
		}
		//查找入职日期的前一个工作日
		TEmployeeInsuranceWorkDayVo newVo = new TEmployeeInsuranceWorkDayVo();
		newVo.setRegistDate(DateUtil.addDayByDate(vo.getRegistDate(), CommonConstants.ONE_INT_NEGATE));
		newVo.setType(CommonConstants.ONE_STRING);
		getWorkDay(holidayMap, newVo);

		//判断前一个工作日是否大于今天
		LocalDate givenLocalDate = newVo.getRegistDate().toInstant()
				.atZone(java.time.ZoneId.systemDefault())
				.toLocalDate();
		// 获取今天的 LocalDate
		LocalDate today = LocalDate.now();
		LocalTime currentTime = LocalTime.now();

		if (givenLocalDate.isBefore(today) || givenLocalDate.isEqual(today)) {
			// 如果日期前一个工作日日期小于等于今天
			if (currentTime.isBefore(LocalTime.of(15, 20)) || currentTime.equals(LocalTime.of(15, 20))) {
				// 如果当前时间在15:20之前
				vo.setPriDisDate(LocalDateTimeUtils.convertLDToDate(today));
				vo.setRegistDate(DateUtil.addDayByDate(vo.getPriDisDate(), CommonConstants.ONE_INT));
			} else {
				// 如果当前时间在15:20之后
				TEmployeeInsuranceWorkDayVo afterVo = new TEmployeeInsuranceWorkDayVo();
				afterVo.setRegistDate(DateUtil.addDayByDate(LocalDateTimeUtils.convertLDToDate(today), CommonConstants.ONE_INT));
				afterVo.setType(CommonConstants.TWO_STRING);
				getWorkDay(holidayMap, afterVo);
				vo.setPriDisDate(afterVo.getRegistDate());
				vo.setRegistDate(DateUtil.addDayByDate(vo.getPriDisDate(), CommonConstants.ONE_INT));
			}
		} else {
			//如果前一个工作日日期大于今天
			//预计派单日期
			vo.setPriDisDate(newVo.getRegistDate());
			//预计生效日期
			vo.setRegistDate(DateUtil.addDayByDate(newVo.getRegistDate(), CommonConstants.ONE_INT));
		}
		return vo;
	}

	@Override
	public Boolean checkIsWorkDay(TEmployeeInsuranceWorkDayVo vo) {
		//假期配置表
		return baseMapper.selectCount(Wrappers.<THolidayInfo>query().lambda()
				.eq(THolidayInfo::getDate,vo.getRegistDate())) > 0;

	}

	private void getWorkDay(Map<Date, String> holidayMap, TEmployeeInsuranceWorkDayVo vo) {
		String day= holidayMap.get(vo.getRegistDate());
		if (Common.isNotNull(day)){
			vo.setRegistDate(DateUtil.addDayByDate(vo.getRegistDate(), vo.getType().equals(CommonConstants.ONE_STRING)
					? CommonConstants.ONE_INT_NEGATE : CommonConstants.ONE_INT));
		}else {
			return;
		}
		getWorkDay(holidayMap,vo);
	}
}
