/*
 * Copyright (c) 2020 yifu4cloud Authors. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.yifu.cloud.plus.v1.yifu.admin.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.pig4cloud.plugin.excel.annotation.RequestExcel;
import com.pig4cloud.plugin.excel.annotation.ResponseExcel;
import com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysRole;
import com.yifu.cloud.plus.v1.yifu.admin.api.vo.HcmRoleInfoVo;
import com.yifu.cloud.plus.v1.yifu.admin.api.vo.RoleExcelVO;
import com.yifu.cloud.plus.v1.yifu.admin.api.vo.RoleVo;
import com.yifu.cloud.plus.v1.yifu.admin.service.SysRoleMenuService;
import com.yifu.cloud.plus.v1.yifu.admin.service.SysRoleService;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.ClientNameConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * @author lengleng
 * @date 2019/2/1
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/role")
@Tag(name = "角色管理模块")
public class RoleController {

	private final SysRoleService sysRoleService;

	private final SysRoleMenuService sysRoleMenuService;

	/**
	 * 通过ID查询角色信息
	 * @param id ID
	 * @return 角色信息
	 */
	@GetMapping("/{id:\\d+}")
	public R<SysRole> getById(@PathVariable Long id) {
		return R.ok(sysRoleService.getById(id));
	}

	/**
	 * 添加角色
	 * @param sysRole 角色信息
	 * @return success、false
	 */
	@SysLog("添加角色")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('sys_role_add')")
	public R<Boolean> save(@Valid @RequestBody SysRole sysRole) {
		R<Boolean> check = sysRoleService.getBooleanR(sysRole);
		if ( Common.isNotNull(check)) {
			return check;
		}
		sysRole.setClientId(ClientNameConstants.CLIENT_MVP);
		return R.ok(sysRoleService.save(sysRole));
	}




	/**
	 * 修改角色
	 * @param sysRole 角色信息
	 * @return success/false
	 */
	@SysLog("修改角色")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('sys_role_edit')")
	public R<Boolean> update(@Valid @RequestBody SysRole sysRole) {
		R<Boolean> check = sysRoleService.getBooleanR(sysRole);
		if ( Common.isNotNull(check)) {
			return check;
		}
		sysRoleService.updateRoleById(sysRole);
		return R.ok();
	}

	/**
	 * 删除角色
	 * @param id
	 * @return
	 */
	@SysLog("删除角色")
	@DeleteMapping("/{id:\\d+}")
	@PreAuthorize("@pms.hasPermission('sys_role_del')")
	public R<Boolean> removeById(@PathVariable Long id) {
		return sysRoleService.removeRoleById(id);
	}

	/**
	 * 获取角色列表
	 * @return 角色列表
	 */
	@GetMapping("/list")
	public R<List<SysRole>> listRoles() {
		return R.ok(sysRoleService.list(Wrappers.<SysRole>query().lambda()
				.eq(SysRole::getClientId,ClientNameConstants.CLIENT_MVP)
				.eq(SysRole::getDelFlag, CommonConstants.ZERO_STRING)));
	}


	/**
	 * 获取角色列表
	 * @return 角色列表
	 */
	@Operation(description = "HCM获取角色列表")
	@SysLog("HCM获取角色列表--HCM调用接口")
	@GetMapping("/getRoleList")
	public R<List<HcmRoleInfoVo>> getRoleList() {
		return R.ok(sysRoleService.getRoleList());
	}

	/**
	 * 分页查询角色信息
	 * @param page 分页对象
	 * @return 分页对象
	 */
	@GetMapping("/page")
	public R<IPage<SysRole>> getRolePage(Page page, SysRole sysRole) {
		sysRole.setClientId(ClientNameConstants.CLIENT_MVP);
		return R.ok(sysRoleService.getSysRoleByPage(page, sysRole));
	}

	/**
	 * 更新角色菜单
	 * @param roleVo 角色对象
	 * @return success、false
	 */
	@SysLog("更新角色菜单")
	@PutMapping("/menu")
	@PreAuthorize("@pms.hasPermission('sys_role_perm')")
	public R<Boolean> saveRoleMenus(@RequestBody RoleVo roleVo) {
		return R.ok(sysRoleMenuService.saveRoleMenus(roleVo.getRoleId(), roleVo.getMenuIds()));
	}

	/**
	 * 导出excel 表格
	 * @return
	 */
	@ResponseExcel
	@GetMapping("/export")
	@PreAuthorize("@pms.hasPermission('sys_role_import_export')")
	public List<RoleExcelVO> export() {
		return sysRoleService.listRole();
	}

	/**
	 * 导入角色
	 * @param excelVOList 角色列表
	 * @param bindingResult 错误信息列表
	 * @return ok fail
	 */
	@PostMapping("/import")
	@PreAuthorize("@pms.hasPermission('sys_role_import_export')")
	public R importRole(@RequestExcel List<RoleExcelVO> excelVOList, BindingResult bindingResult) {
		return sysRoleService.importRole(excelVOList, bindingResult);
	}

}
