/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TCompleteMonitorSet;
import com.yifu.cloud.plus.v1.yifu.archives.service.TCompleteMonitorSetService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.TCompleteMonitorSetSearchVo;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;


/**
 * 完整度监控配置
 *
 * @author hgw
 * @date 2025-02-26 15:18:51
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tcompletemonitorset")
@Tag(name = "完整度监控配置管理")
public class TCompleteMonitorSetController {

	private final TCompleteMonitorSetService tCompleteMonitorSetService;


	/**
	 * 简单分页查询
	 *
	 * @param page                分页对象
	 * @param tCompleteMonitorSet 完整度监控配置
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TCompleteMonitorSet>> getTCompleteMonitorSetPage(Page<TCompleteMonitorSet> page, TCompleteMonitorSetSearchVo tCompleteMonitorSet) {
		return new R<>(tCompleteMonitorSetService.getTCompleteMonitorSetPage(page, tCompleteMonitorSet));
	}

	/**
	 * 不分页查询
	 *
	 * @param tCompleteMonitorSet 完整度监控配置
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	public R<List<TCompleteMonitorSet>> getTCompleteMonitorSetNoPage(@RequestBody TCompleteMonitorSetSearchVo tCompleteMonitorSet) {
		return R.ok(tCompleteMonitorSetService.noPageDiy(tCompleteMonitorSet));
	}

	/**
	 * 通过id查询完整度监控配置
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	public R<TCompleteMonitorSet> getById(@PathVariable("id") String id) {
		return R.ok(tCompleteMonitorSetService.getById(id));
	}

	/**
	 * @Description: 获取可用的员工合同类型
	 * @Author: hgw
	 * @Date: 2025/2/27 15:24
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysDictItem>>
	 **/
	@Operation(summary = "获取在用的员工合同类型", description = "获取在用的员工合同类型")
	@GetMapping("/getContractType")
	public R<String> getContractType() {
		return R.ok(tCompleteMonitorSetService.getContractType());
	}

	/**
	 * 新增完整度监控配置
	 *
	 * @param tCompleteMonitorSet 完整度监控配置
	 * @return R
	 */
	@Operation(summary = "新增完整度监控配置", description = "新增完整度监控配置：hasPermission('archives_tcompletemonitorset_add')")
	@SysLog("新增完整度监控配置")
	@PostMapping("/saveInfo")
	@PreAuthorize("@pms.hasPermission('archives_tcompletemonitorset_add')")
	public R<String> saveInfo(@RequestBody TCompleteMonitorSet tCompleteMonitorSet) {
		return tCompleteMonitorSetService.saveOrUpdateInfo(tCompleteMonitorSet);
	}

	/**
	 * 修改完整度监控配置
	 *
	 * @param tCompleteMonitorSet 完整度监控配置
	 * @return R
	 */
	@Operation(summary = "修改完整度监控配置", description = "修改完整度监控配置：hasPermission('archives_tcompletemonitorset_edit')")
	@SysLog("修改完整度监控配置")
	@PostMapping("/updateInfoById")
	@PreAuthorize("@pms.hasPermission('archives_tcompletemonitorset_edit')")
	public R<String> updateInfoById(@RequestBody TCompleteMonitorSet tCompleteMonitorSet) {
		if (Common.isEmpty(tCompleteMonitorSet.getId())) {
			return R.failed("请传Id");
		}
		return tCompleteMonitorSetService.saveOrUpdateInfo(tCompleteMonitorSet);
	}

	/**
	 * 通过id删除完整度监控配置
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除完整度监控配置", description = "通过id删除完整度监控配置：hasPermission('archives_tcompletemonitorset_del')")
	@SysLog("通过id删除完整度监控配置")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('archives_tcompletemonitorset_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tCompleteMonitorSetService.removeById(id));
	}

}
