/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.social.entity.TSocialFreindPushAndGet;
import com.yifu.cloud.plus.v1.yifu.social.service.TSocialFreindPushAndGetService;
import com.yifu.cloud.plus.v1.yifu.social.service.TSocialFriendPushService;
import com.yifu.cloud.plus.v1.yifu.social.service.TSocialFriendService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.concurrent.atomic.AtomicInteger;


/**
 * 税友
 *
 * @author hgw
 * @date 2024-12-6 15:46:04
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tsocialfriend")
@Tag(name = "税友")
public class TSocialFriendController {

	private AtomicInteger atomicPush = new AtomicInteger(0);
	private AtomicInteger atomicGet = new AtomicInteger(0);

	private final TSocialFriendService tSocialFriendService;
	private final TSocialFriendPushService tSocialFriendPushService;
	private final TSocialFreindPushAndGetService tSocialFreindPushAndGetService;

	@Operation(description = "测试--测试签名使用暂时注释")
	@PostMapping("/doTest")
	@SysLog("测试")
	public R<String> test() {
		return tSocialFriendPushService.test();
	}


	@Operation(description = "1定时任务推送税友")
	@PostMapping("/inner/doPushFriend")
	@SysLog("1定时任务推送税友")
	@Inner
	public R<String> doPushFriendTest() {
		return tSocialFriendPushService.pushFriend(null);
	}

	@Operation(description = "2定时任务获取税友")
	@PostMapping("/inner/doGetFriendByRequestId")
	@SysLog("2定时任务获取税友")
	@Inner
	public R<String> doGetFriendByRequestId() {
		return tSocialFriendService.getInfoByRequestId();
	}

	/**
	 * @Description: 主动推送税友
	 * @Author: hgw
	 * @Date: 2025/3/18 10:44
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(description = "主动推送税友")
	@GetMapping("/doPushFriendByActive")
	@SysLog("主动推送税友")
	public R<String> doPushFriendActive() {
		if (atomicPush.incrementAndGet() <= 1) {
			try {
				R<String> failed = getCore(CommonConstants.ONE_INT);
				if (failed != null) return failed;
			} finally {
				atomicPush.decrementAndGet();
			}
			return R.ok();
		} else {
			atomicPush.decrementAndGet();
			return R.failed("距离上一次推送任务时间间隔太短，请10分钟后再试！！");
		}
	}

	/**
	 * @param type 1推送2拉取
	 * @Description: 核心处理
	 * @Author: hgw
	 * @Date: 2025/3/18 17:27
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	private R<String> getCore(int type) {
		YifuUser user = SecurityUtils.getUser();
		if (user == null || Common.isEmpty(user.getNickname())) {
			return R.failed("请先登录！");
		}
		// 先加限制 查有没有到10分钟
		// 获取当前时间
		LocalDateTime now = LocalDateTime.now();
		// 计算10分钟前的时间
		LocalDateTime tenMinutesAgo = now.minusMinutes(10);
		LocalDateTime time = tSocialFreindPushAndGetService.getMaxTimeByType(type);
		if (Common.isEmpty(time) || time.isBefore(tenMinutesAgo)) {
			TSocialFreindPushAndGet info = new TSocialFreindPushAndGet();
			info.setType(type);
			info.setCreateTime(now);
			info.setCreateUserName(user.getNickname());
			tSocialFreindPushAndGetService.save(info);
			if (type == CommonConstants.ONE_INT) {
				//doJointFriendTask.pushFriendByAsync()
				tSocialFriendPushService.pushFriend(null);
			} else {
				//doJointFriendTask.getInfoByRequestIdByAsync()
				tSocialFriendService.getInfoByRequestId();
			}
			return R.ok();
		}
		if (Common.isNotNull(time) && time.isAfter(tenMinutesAgo)) {
			if (type == CommonConstants.ONE_INT) {
				return R.failed("距离上一次推送任务时间间隔太短，请10分钟后再试！");
			} else {
				return R.failed("距离上一次拉取任务时间间隔太短，请10分钟后再试！");
			}
		}
		return null;
	}

	/**
	 * @Description: 主动获取税友
	 * @Author: hgw
	 * @Date: 2025/3/18 10:45
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(description = "主动获取税友")
	@GetMapping("/doGetFriendByActive")
	@SysLog("主动获取税友")
	public R<String> doGetFriendByActive() {
		if (atomicGet.incrementAndGet() <= 1) {
			try {
				R<String> failed = getCore(CommonConstants.TWO_INT);
				if (failed != null) return failed;
			} finally {
				atomicGet.decrementAndGet();
			}
			return R.ok();
		} else {
			atomicGet.decrementAndGet();
			return R.failed("距离上一次推送任务时间间隔太短，请10分钟后再试！！");
		}
	}

	/**
	 * @Author fxj
	 * @Description 定时任务每月1日3点定时更新跨月的继续办理的数据为终止办理
	 * @Date 10:40 2025/7/14
	 * @Param
	 * @return
	**/
	@Operation(description = "定时任务每月1日3点定时更新跨月的继续办理的数据为终止办理")
	@PostMapping("/inner/doChangeHandleStatusToEnd")
	@SysLog("定时任务每月1日3点定时更新跨月的继续办理的数据为终止办理")
	@Inner
	public void doChangeHandleStatusToEnd() {
		tSocialFriendService.changeSocialHandndleStatus();
	}
}
