/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.social.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ArrayUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.read.listener.ReadListener;
import com.alibaba.excel.read.metadata.holder.ReadRowHolder;
import com.alibaba.excel.util.ListUtils;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.*;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.mybatis.base.BaseEntity;
import com.yifu.cloud.plus.v1.yifu.social.entity.SysHouseHoldInfo;
import com.yifu.cloud.plus.v1.yifu.social.entity.TDispatchInfo;
import com.yifu.cloud.plus.v1.yifu.social.entity.TSocialInfo;
import com.yifu.cloud.plus.v1.yifu.social.mapper.SysHouseHoldInfoMapper;
import com.yifu.cloud.plus.v1.yifu.social.mapper.TDispatchInfoMapper;
import com.yifu.cloud.plus.v1.yifu.social.mapper.TSocialInfoMapper;
import com.yifu.cloud.plus.v1.yifu.social.service.TAuditInfoService;
import com.yifu.cloud.plus.v1.yifu.social.service.TSocialInfoService;
import com.yifu.cloud.plus.v1.yifu.social.vo.SocialAddAndReduceVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.TSocialInfoSearchVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.TSocialInfoVo;
import lombok.AllArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
/**
 * 社保明细表
 *
 * @author fxj
 * @date 2022-07-15 11:38:05
 */
@AllArgsConstructor
@Log4j2
@Service
public class TSocialInfoServiceImpl extends ServiceImpl<TSocialInfoMapper, TSocialInfo> implements TSocialInfoService {

	private final TDispatchInfoMapper dispatchInfoMapper;

	private final SysHouseHoldInfoMapper sysHouseHoldInfoMapper;

	private final TAuditInfoService tAuditInfoService;

    /**
     * 社保明细表简单分页查询
     * @param tSocialInfo 社保明细表
     * @return
     */
    @Override
    public IPage<TSocialInfo> getTSocialInfoPage(Page<TSocialInfo> page, TSocialInfo tSocialInfo){
        return baseMapper.getTSocialInfoPage(page,tSocialInfo);
    }
    /**
     * 社保明细表批量导出
     * @return
     */
    @Override
    public void listExport(HttpServletResponse response, TSocialInfoSearchVo searchVo){
        String fileName = "不良记录批量导出" + DateUtil.getThisTime() + CommonConstants.XLSX;
        //获取要导出的列表
        List<TSocialInfo> list = new ArrayList<>();
        long count = noPageCountDiy(searchVo);
        ServletOutputStream out = null;
        try {
            out = response.getOutputStream();
            response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
            response.setCharacterEncoding("utf-8");
            response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName  , "UTF-8"));
            // 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
            //EasyExcel.write(out, TEmpBadRecord.class).sheet("不良记录").doWrite(list);
            ExcelWriter  excelWriter = EasyExcel.write(out, TSocialInfo.class).build();
            int index = 0;
            if (count >  CommonConstants.ZERO_INT){
				WriteSheet writeSheet;
				ExcelUtil<TSocialInfo> util;
                for (int i = 0; i <= count; ) {
                    // 获取实际记录
                    searchVo.setLimitStart(i);
                    searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
                    list = noPageDiy(searchVo);
                    if (Common.isNotNull(list)){
                        util = new ExcelUtil<>(TSocialInfo.class);
                        for (TSocialInfo vo:list){
                            util.convertEntity(vo,null,null,null);
                        }
                    }
                    if (Common.isNotNull(list)){
                        writeSheet = EasyExcel.writerSheet("社保明细表"+index).build();
                        excelWriter.write(list,writeSheet);
                        index++;
                    }
                    i = i + CommonConstants.EXCEL_EXPORT_LIMIT;
                    if (Common.isNotNull(list)){
                        list.clear();
                    }
                }
            }else {
                WriteSheet writeSheet = EasyExcel.writerSheet("社保明细表"+index).build();
                excelWriter.write(list,writeSheet);
            }
			if (Common.isNotNull(list)){
                list.clear();
            }
            out.flush();
            excelWriter.finish();
        }catch (Exception e){
            log.error("执行异常" ,e);
        }finally {
            try {
                if (null != out) {
                    out.close();
                }
            } catch (IOException e) {
                log.error("执行异常", e);
            }
        }
    }

    @Override
    public List<TSocialInfo> noPageDiy(TSocialInfoSearchVo searchVo) {
        LambdaQueryWrapper<TSocialInfo> wrapper = buildQueryWrapper(searchVo);
        List<String> idList = Common.getList(searchVo.getIds());
        if (Common.isNotNull(idList)){
            wrapper.in(TSocialInfo::getId,idList);
        }
        if (searchVo.getLimitStart() >= 0 && searchVo.getLimitEnd() > 0){
            wrapper.last(" limit "+ searchVo.getLimitStart() +","+ searchVo.getLimitEnd());
        }
        wrapper.orderByDesc(BaseEntity::getCreateTime);
        return baseMapper.selectList(wrapper);
    }

    private  Long noPageCountDiy(TSocialInfoSearchVo searchVo) {
        LambdaQueryWrapper<TSocialInfo> wrapper = buildQueryWrapper(searchVo);
        List<String> idList = Common.getList(searchVo.getIds());
        if (Common.isNotNull(idList)){
            wrapper.in(TSocialInfo::getId,idList);
        }
        return baseMapper.selectCount(wrapper);
    }

    private LambdaQueryWrapper buildQueryWrapper(TSocialInfoSearchVo entity){
        LambdaQueryWrapper<TSocialInfo> wrapper = Wrappers.lambdaQuery();
        if (ArrayUtil.isNotEmpty(entity.getCreateTimes())) {
            wrapper.ge(TSocialInfo::getCreateTime, entity.getCreateTimes()[0])
                    .le(TSocialInfo::getCreateTime,
                            entity.getCreateTimes()[1]);
        }
        if (Common.isNotNull(entity.getCreateName())){
            wrapper.eq(TSocialInfo::getCreateName,entity.getCreateName());
        }
        return wrapper;
    }

    @Override
    public R<List<ErrorMessage>> importDiy(InputStream inputStream) {
        List<ErrorMessage> errorMessageList = new ArrayList<>();
        ExcelUtil<TSocialInfoVo> util1 = new ExcelUtil<>(TSocialInfoVo.class);
        // 写法2：
        // 匿名内部类 不用额外写一个DemoDataListener
        // 这里 需要指定读用哪个class去读，然后读取第一个sheet 文件流会自动关闭
        try {
            EasyExcel.read(inputStream, TSocialInfoVo.class, new ReadListener<TSocialInfoVo>() {
                /**
                 * 单次缓存的数据量
                 */
                public static final int BATCH_COUNT = CommonConstants.BATCH_COUNT;
                /**
                 *临时存储
                 */
                private List<TSocialInfoVo> cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);

                @Override
                public void invoke(TSocialInfoVo data, AnalysisContext context) {
                    ReadRowHolder readRowHolder = context.readRowHolder();
                    Integer rowIndex = readRowHolder.getRowIndex();
                    data.setRowIndex(rowIndex+1);
                    ErrorMessage errorMessage = util1.checkEntity(data, data.getRowIndex());
                    if (Common.isNotNull(errorMessage)){
                        errorMessageList.add(errorMessage);
                    }else {
                        cachedDataList.add(data);
                    }
                    if (cachedDataList.size() >= BATCH_COUNT) {
                        saveData();
                        // 存储完成清理 list
                        cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);
                    }
                }
                @Override
                public void doAfterAllAnalysed(AnalysisContext context) {
                    saveData();
                }

                /**
                 * 加上存储数据库
                 */
                private void saveData() {
                    log.info("{}条数据，开始存储数据库！", cachedDataList.size());
                    importTSocialInfo(cachedDataList,errorMessageList);
                    log.info("存储数据库成功！");
                }
            }).sheet().doRead();
        }catch (Exception e){
            log.error(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR,e);
            return R.failed(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR);
        }
        return  R.ok(errorMessageList);
    }
    private void importTSocialInfo(List<TSocialInfoVo> excelVOList, List<ErrorMessage> errorMessageList) {
        // 个性化校验逻辑
        // 执行数据插入操作 组装
        for (int i = 0; i < excelVOList.size(); i++) {
            TSocialInfoVo excel = excelVOList.get(i);
            // 数据合法情况 TODO

            // 插入
            insertExcel(excel);
            errorMessageList.add(new ErrorMessage(excel.getRowIndex(),CommonConstants.SAVE_SUCCESS));
        }
    }
    /**
     * 插入excel bad record
     */
    private void insertExcel(TSocialInfoVo excel) {
        TSocialInfo insert = new TSocialInfo();
        BeanUtil.copyProperties(excel, insert);
        this.save(insert);
    }

	/**
	 * 通过ID查询员工对应的派增派减数据
	 * @Author fxj
	 * @Date 2020-08-25
	 * @param empId
	 * @return
	 **/
	@Override
	public SocialAddAndReduceVo getAddOrReduceInfoByEmpId(String empId) {
		SocialAddAndReduceVo socialAddAndReduceVo = new SocialAddAndReduceVo();
		TSocialInfo socialInfo;
		TDispatchInfo dispatchInfoAdd = dispatchInfoMapper.selectOne(Wrappers.<TDispatchInfo>query().lambda()
				.eq(TDispatchInfo::getEmpId,empId)
				.eq(TDispatchInfo::getDeleteFlag,CommonConstants.ZERO_STRING)
				.eq(TDispatchInfo::getType,CommonConstants.ZERO_STRING)
				.isNotNull(TDispatchInfo::getSocialId)
				.orderByDesc(TDispatchInfo::getCreateTime)
				.last(CommonConstants.LAST_ONE_SQL));
		if (null != dispatchInfoAdd){
			socialInfo = baseMapper.selectById(dispatchInfoAdd.getSocialId());
			initSocialAddAndReduce(socialAddAndReduceVo, socialInfo, dispatchInfoAdd);
			if (null != dispatchInfoAdd.getCreateTime()){
				socialAddAndReduceVo.setSocialAddDispatchDate(LocalDateTimeUtils.formatTime(dispatchInfoAdd.getCreateTime(),LocalDateTimeUtils.DATE_TIME_PATTERN_DEFAULT));
			}
		}
		TDispatchInfo dispatchInfoReduce = dispatchInfoMapper.selectOne(Wrappers.<TDispatchInfo>query().lambda()
				.eq(TDispatchInfo::getEmpId,empId)
				.eq(TDispatchInfo::getDeleteFlag,CommonConstants.ZERO_STRING)
				.eq(TDispatchInfo::getType,CommonConstants.ONE_STRING)
				.isNotNull(TDispatchInfo::getSocialId)
				.orderByDesc(TDispatchInfo::getCreateTime)
				.last(CommonConstants.LAST_ONE_SQL));
		if (null != dispatchInfoReduce){
			socialInfo = baseMapper.selectById(dispatchInfoReduce.getSocialId());
			initSocialAddAndReduce(socialAddAndReduceVo, socialInfo, dispatchInfoReduce);
			if (Common.isNotNull(dispatchInfoReduce.getSocialReduceDate())) {
				socialAddAndReduceVo.setSocialEndDate(DateUtil.formatDate(dispatchInfoReduce.getSocialReduceDate()));
			}
			if (null != dispatchInfoReduce.getCreateTime()){
				socialAddAndReduceVo.setSocialReduceDispatchDate(LocalDateTimeUtils.formatTime(dispatchInfoReduce.getCreateTime(),LocalDateTimeUtils.DATE_TIME_PATTERN_DEFAULT));
			}
		}
		return socialAddAndReduceVo;
	}

	/**
	 * 初始化社保派增派减数据
	 * @Author fxj
	 * @Date 2020-08-25
	 * @param socialAddAndReduceVo
	 * @param socialInfo
	 * @param dispatchInfo
	 * @return
	 **/
	private void initSocialAddAndReduce(SocialAddAndReduceVo socialAddAndReduceVo, TSocialInfo socialInfo, TDispatchInfo dispatchInfo) {
		// 初始化户和区域数据
		initSocialHouseAndAddress(socialAddAndReduceVo, socialInfo,dispatchInfo.getType());
		// 派增派单状态处理 待提交和待审核都属于派单开始
		if (CommonConstants.ZERO_STRING.equals(dispatchInfo.getType())){
			initSocialAddStatus(socialAddAndReduceVo, dispatchInfo);
			// 派减派单状态处理 待提交和待审核都属于派单开始
		}else {
			initSocialReduceStatus(socialAddAndReduceVo, dispatchInfo);
		}
	}

	/**
	 * 初始化派增派减地址
	 * @Author fxj
	 * @Date 2020-08-25
	 * @param socialAddAndReduceVo
	 * @param socialInfo
	 * @param type
	 * @param houseHoldInfo
	 * @return
	 **/
	private void initSocialAddress(SocialAddAndReduceVo socialAddAndReduceVo, TSocialInfo socialInfo, String type, SysHouseHoldInfo houseHoldInfo) {
		if (CommonConstants.ZERO_STRING.equals(type)){
			socialAddAndReduceVo.setSocialHouse(null == houseHoldInfo?"":houseHoldInfo.getName());
			socialAddAndReduceVo.setSocialProvince(Common.isNotNull(socialInfo.getSocialProvince())? Integer.parseInt(socialInfo.getSocialProvince()):null);
			socialAddAndReduceVo.setSocialCity(Common.isNotNull(socialInfo.getSocialCity())? Integer.parseInt(socialInfo.getSocialCity()):null);
			socialAddAndReduceVo.setSocialTown(Common.isNotNull(socialInfo.getSocialTown())? Integer.parseInt(socialInfo.getSocialTown()):null);
			if (null != socialInfo.getSocialStartDate()) {
				socialAddAndReduceVo.setSocialStartDate(DateUtil.formatDate(socialInfo.getSocialStartDate()));
			}
		}else {
			socialAddAndReduceVo.setSocialHouseReduce(null==houseHoldInfo?"":houseHoldInfo.getName());
			socialAddAndReduceVo.setSocialProvinceReduce(Common.isNotNull(socialInfo.getSocialProvince())? Integer.parseInt(socialInfo.getSocialProvince()):null);
			socialAddAndReduceVo.setSocialCityReduce(Common.isNotNull(socialInfo.getSocialCity())? Integer.parseInt(socialInfo.getSocialCity()):null);
			socialAddAndReduceVo.setSocialTownReduce(Common.isNotNull(socialInfo.getSocialTown())? Integer.parseInt(socialInfo.getSocialTown()):null);
			if (null != socialInfo.getSocialStartDate()) {
				socialAddAndReduceVo.setSocialStartDate(DateUtil.formatDate(socialInfo.getSocialStartDate()));
			}
		}
	}

	/**
	 * 初始化社保派减办理状态
	 * @Author fxj
	 * @Date 2020-08-25
	 * @param socialAddAndReduceVo
	 * @param dispatchInfo
	 * @return
	 **/
	private void initSocialAddStatus(SocialAddAndReduceVo socialAddAndReduceVo, TDispatchInfo dispatchInfo) {
		//new:社保派增办理状态  0 派单开始  1  办理中 2 办理成功 3 办理失败 4 部分办理成功
		if (CommonConstants.ZERO_STRING.equals(dispatchInfo.getStatus())
				|| CommonConstants.ONE_STRING.equals(dispatchInfo.getStatus())
				|| CommonConstants.THREE_STRING.equals(dispatchInfo.getStatus())) {
			socialAddAndReduceVo.setSocialAddStatus(CommonConstants.ZERO_STRING);
			// 审核不通过和 审核通过 都属于办理中
		} else if (CommonConstants.TWO_STRING.equals(dispatchInfo.getStatus())) {
			socialAddAndReduceVo.setSocialAddStatus(CommonConstants.ONE_STRING);
		}
		// 办理成功
		if (CommonConstants.ONE_STRING.equals(dispatchInfo.getSocialHandleStatus())) {
			socialAddAndReduceVo.setSocialAddStatus(CommonConstants.TWO_STRING);
		}
		// 办理失败
		if (CommonConstants.TWO_STRING.equals(dispatchInfo.getSocialHandleStatus())) {
			socialAddAndReduceVo.setSocialAddStatus(CommonConstants.THREE_STRING);
		}
		// 部分办理失败
		if (CommonConstants.THREE_STRING.equals(dispatchInfo.getSocialHandleStatus())) {
			socialAddAndReduceVo.setSocialAddStatus(CommonConstants.FOUR_STRING);
		}
	}
	/**
	 * 初始化社保户和社保缴纳地
	 * @Author fxj
	 * @Date 2020-08-25
	 * @param socialAddAndReduceVo
	 * @param socialInfo
	 * @param type
	 * @return
	 **/
	private void initSocialHouseAndAddress(SocialAddAndReduceVo socialAddAndReduceVo, TSocialInfo socialInfo, String type) {
		if (null != socialInfo) {
			SysHouseHoldInfo houseHoldInfo = null;
			if (null != socialInfo.getSocialHousehold()) {
				houseHoldInfo = sysHouseHoldInfoMapper.selectById(socialInfo.getSocialHousehold());
			}
			initSocialAddress(socialAddAndReduceVo, socialInfo, type, houseHoldInfo);
		}
	}

	/**
	 * 初始化派增办理状态
	 * @Author fxj
	 * @Date 2020-08-25
	 * @param socialAddAndReduceVo
	 * @param dispatchInfo
	 * @return
	 **/
	private void initSocialReduceStatus(SocialAddAndReduceVo socialAddAndReduceVo, TDispatchInfo dispatchInfo) {
		// 社保派减办理状态 0 派单开始  1  办理中 2 办理成功 3 办理失败
		if (CommonConstants.ZERO_STRING.equals(dispatchInfo.getStatus())
				|| CommonConstants.ONE_STRING.equals(dispatchInfo.getStatus())
				|| CommonConstants.THREE_STRING.equals(dispatchInfo.getStatus())) {
			socialAddAndReduceVo.setSocialReduceStatus(CommonConstants.ZERO_STRING);
			// 审核不通过和 审核通过 都属于办理中
		} else if (CommonConstants.TWO_STRING.equals(dispatchInfo.getStatus())) {
			socialAddAndReduceVo.setSocialReduceStatus(CommonConstants.ONE_STRING);
		}
		// 办理成功
		if (CommonConstants.ONE_STRING.equals(dispatchInfo.getSocialHandleStatus())) {
			socialAddAndReduceVo.setSocialReduceStatus(CommonConstants.TWO_STRING);
		}
		// 办理失败
		if (CommonConstants.TWO_STRING.equals(dispatchInfo.getSocialHandleStatus())) {
			socialAddAndReduceVo.setSocialReduceStatus(CommonConstants.THREE_STRING);
		}
	}

	/**
	 * @param
	 * @Description: 获取所有需要自动办理的社保
	 * @Author: hgw
	 * @Date: 2024/5/11 14:55
	 * @return: java.util.List<com.yifu.cloud.plus.v1.yifu.social.entity.TSocialInfo>
	 **/
	@Override
	public List<TSocialInfo> getSocialSoldierYgsAll() {
		return baseMapper.getSocialSoldierYgsAll();
	}
	/**
	 * @param
	 * @Description: 获取所有需要自动办理的社保
	 * @Author: hgw
	 * @Date: 2024/5/11 14:55
	 * @return: java.util.List<com.yifu.cloud.plus.v1.yifu.social.entity.TSocialInfo>
	 **/
	@Override
	public List<TSocialInfo> getSocialSoldierYsdAll() {
		return baseMapper.getSocialSoldierYsdAll();
	}

	@Override
	public List<TSocialInfo> getSocialFriendYgsAll() {
		return baseMapper.getSocialFriendYgsAll();
	}
	/**
	 * @param
	 * @Description: 获取所有需要自动办理的社保
	 * @Author: hgw
	 * @Date: 2024/5/11 14:55
	 * @return: java.util.List<com.yifu.cloud.plus.v1.yifu.social.entity.TSocialInfo>
	 **/
	@Override
	public List<TSocialInfo> getSocialFriendYsdAll() {
		return baseMapper.getSocialFriendYsdAll();
	}
	/**
	 * @param
	 * @Description: 获取所有需要社保局审核的社保
	 * @Author: hgw
	 * @Date: 2024/5/11 14:55
	 * @return: java.util.List<com.yifu.cloud.plus.v1.yifu.social.entity.TSocialInfo>
	 **/
	@Override
	public List<TSocialInfo> getSocialSoldierYgsByAudit() {
		return baseMapper.getSocialSoldierYgsByAudit();
	}
	/**
	 * @param
	 * @Description: 获取所有需要社保局审核的社保
	 * @Author: hgw
	 * @Date: 2024/5/11 14:55
	 * @return: java.util.List<com.yifu.cloud.plus.v1.yifu.social.entity.TSocialInfo>
	 **/
	@Override
	public List<TSocialInfo> getSocialSoldierYsdByAudit() {
		return baseMapper.getSocialSoldierYsdByAudit();
	}

	/**
	 * @param
	 * @Description: 根据任务ID，获取需要自动办理的社保
	 * @Author: hgw
	 * @Date: 2024/5/11 14:55
	 * @return: java.util.List<com.yifu.cloud.plus.v1.yifu.social.entity.TSocialInfo>
	 **/
	@Override
	public List<TSocialInfo> getSocialSoldierByAddId(String addId) {
		return baseMapper.getSocialSoldierByAddId(addId);
	}

	/**
	 * @param idList 社保IDlist
	 * @param isSocial 1：社保 2：医保 3：全部
	 * @param user 当前登录人
	 * @Description: 批量转人工
	 * @Author: hgw
	 * @Date: 2025/4/25 17:24
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Override
	public R<String> doRenGongBatch(List<String> idList, String isSocial, YifuUser user) {
		for (String id : idList) {
			doRenGong(id, isSocial, user);
		}
		return R.ok("成功！");
	}

	/**
	 * @param id 社保ID
	 * @param isSocial 1：社保 2：医保 3：全部
	 * @param user 当前登录人
	 * @Description: 转人工
	 * @Author: hgw
	 * @Date: 2025/4/25 17:24
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Override
	public R<String> doRenGong(String id, String isSocial, YifuUser user) {
		TSocialInfo socialInfo = this.getById(id);
		if (socialInfo == null || Common.isEmpty(socialInfo.getId())) {
			return R.failed("未找到社保");
		} else {
			String handleRemark;
			// 有可以转的项目，则可以转
			boolean canUpdate = false;
			if (CommonConstants.ZERO_STRING.equals(socialInfo.getDispatchType())) {
				handleRemark = "派增";
			} else {
				handleRemark = "派减";
			}
			// 判断养工失转人工
			if ((CommonConstants.ONE_STRING.equals(isSocial) || CommonConstants.THREE_STRING.equals(isSocial))
					&& Common.isNotNull(socialInfo.getYgsHandleStatus())) {
				if (CommonConstants.ZERO_STRING.equals(socialInfo.getYgsHandleStatus())
						|| CommonConstants.FOUR_STRING.equals(socialInfo.getYgsHandleStatus())
						|| CommonConstants.FIVE_STRING.equals(socialInfo.getYgsHandleStatus())
						|| CommonConstants.SIX_STRING.equals(socialInfo.getYgsHandleStatus())) {
					// 全部的，不反馈某一个失败
					if (!CommonConstants.THREE_STRING.equals(isSocial)) {
						return R.failed("当前养老工伤失业-办理状态不可以转人工！");
					}
				} else {
					socialInfo.setYgsHandleStatus(CommonConstants.FIVE_STRING);
					handleRemark += "养老、工伤、失业手动转人工;";
					canUpdate = true;
				}
			}
			// 判断医生大转人工
			if ((CommonConstants.TWO_STRING.equals(isSocial) || CommonConstants.THREE_STRING.equals(isSocial))
					&& Common.isNotNull(socialInfo.getYsdHandleStatus())) {
				if (CommonConstants.ZERO_STRING.equals(socialInfo.getYsdHandleStatus())
						|| CommonConstants.FOUR_STRING.equals(socialInfo.getYsdHandleStatus())
						|| CommonConstants.FIVE_STRING.equals(socialInfo.getYsdHandleStatus())
						|| CommonConstants.SIX_STRING.equals(socialInfo.getYsdHandleStatus())) {
					// 全部的，不反馈某一个失败
					if (!CommonConstants.THREE_STRING.equals(isSocial)) {
						return R.failed("当前医疗生育大病-办理状态不可以转人工！");
					}
				} else {
					socialInfo.setYsdHandleStatus(CommonConstants.FIVE_STRING);
					handleRemark += "医疗、生育、大病手动转人工";
					canUpdate = true;
				}
			}
			// 更新社保和加日志
			return doUpdateCoreAndLog(id, user, socialInfo, handleRemark, canUpdate);
		}
	}

	/**
	 * @param idList 社保IDlist
	 * @param isSocial 1：社保 2：医保 3：全部
	 * @param user 当前登录人
	 * @Description: 批量转自动化
	 * @Author: hgw
	 * @Date: 2025/4/25 17:24
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Override
	public R<String> doAutoBatch(List<String> idList, String isSocial, YifuUser user) {
		for (String id : idList) {
			doAuto(id, isSocial, user);
		}
		return R.ok("成功！");
	}

	/**
	 * @param id 社保ID
	 * @param isSocial 1：社保 2：医保 3：全部
	 * @param user 当前登录人
	 * @Description: 转自动化
	 * @Author: hgw
	 * @Date: 2025/4/25 17:24
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Override
	public R<String> doAuto(String id, String isSocial, YifuUser user) {
		TSocialInfo socialInfo = this.getById(id);
		if (socialInfo == null || Common.isEmpty(socialInfo.getId())) {
			return R.failed("未找到社保");
		} else {
			String handleRemark;
			// 有可以转的项目，则可以转
			boolean canUpdate = false;
			if (CommonConstants.ZERO_STRING.equals(socialInfo.getDispatchType())) {
				handleRemark = "派增";
			} else {
				handleRemark = "派减";
			}
			// 判断养工失转自动化
			if ((CommonConstants.ONE_STRING.equals(isSocial) || CommonConstants.THREE_STRING.equals(isSocial))
					&& Common.isNotNull(socialInfo.getYgsHandleStatus())
					&& CommonConstants.FIVE_STRING.equals(socialInfo.getYgsHandleStatus())) {
				// 养工失状态为待办理或派减待办理
				if (CommonConstants.ZERO_STRING.equals(socialInfo.getPensionHandle())
						|| CommonConstants.FIVE_STRING.equals(socialInfo.getPensionHandle())

						|| CommonConstants.ZERO_STRING.equals(socialInfo.getWorkInjuryHandle())
						|| CommonConstants.FIVE_STRING.equals(socialInfo.getWorkInjuryHandle())

						|| CommonConstants.ZERO_STRING.equals(socialInfo.getUnemployHandle())
						|| CommonConstants.FIVE_STRING.equals(socialInfo.getUnemployHandle())) {
					socialInfo.setYgsHandleStatus(CommonConstants.ONE_STRING);
					handleRemark += "养老、工伤、失业手动转自动化;";
					canUpdate = true;
				} else if (!CommonConstants.THREE_STRING.equals(isSocial)) {
					// 全部的，不反馈某一个失败
					return R.failed("当前养老工伤失业-办理状态不可以转自动化！");
				}
			}
			// 判断医生大转自动化
			if ((CommonConstants.TWO_STRING.equals(isSocial) || CommonConstants.THREE_STRING.equals(isSocial))
					&& Common.isNotNull(socialInfo.getYsdHandleStatus())
					&& CommonConstants.FIVE_STRING.equals(socialInfo.getYsdHandleStatus())) {
				// 医生大状态为待办理或派减待办理
				if (CommonConstants.ZERO_STRING.equals(socialInfo.getPensionHandle())
						|| CommonConstants.FIVE_STRING.equals(socialInfo.getPensionHandle())

						|| CommonConstants.ZERO_STRING.equals(socialInfo.getWorkInjuryHandle())
						|| CommonConstants.FIVE_STRING.equals(socialInfo.getWorkInjuryHandle())

						|| CommonConstants.ZERO_STRING.equals(socialInfo.getUnemployHandle())
						|| CommonConstants.FIVE_STRING.equals(socialInfo.getUnemployHandle())) {
					socialInfo.setYsdHandleStatus(CommonConstants.FIVE_STRING);
					handleRemark += "医疗、生育、大病手动转自动化";
					canUpdate = true;
				} else if (!CommonConstants.THREE_STRING.equals(isSocial)) {
					// 全部的，不反馈某一个失败
					return R.failed("当前医疗生育大病-办理状态不可以转自动化！");
				}
			}
			// 更新社保和加日志
			return doUpdateCoreAndLog(id, user, socialInfo, handleRemark, canUpdate);
		}
	}

	/**
	 * @param id 社保ID
	 * @param user 当前登录人
	 * @param socialInfo 社保信息
	 * @param handleRemark 操作备注
	 * @param canUpdate 是否可以更新
	 * @Description: 更新社保和加日志
	 * @Author: hgw
	 * @Date: 2025/4/25 17:24
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	private R<String> doUpdateCoreAndLog(String id, YifuUser user, TSocialInfo socialInfo, String handleRemark, boolean canUpdate) {
		if (canUpdate) {
			this.updateById(socialInfo);
			TDispatchInfo disExist = dispatchInfoMapper.selectOne(Wrappers.<TDispatchInfo>query().lambda()
					.eq(TDispatchInfo::getSocialId, id).last(CommonConstants.LAST_ONE_SQL));
			if (disExist != null) {
				socialInfo.setDispatchId(disExist.getId());
			}
			// 加流程进展明细
			tAuditInfoService.doSaveAuditInfo(socialInfo, handleRemark, handleRemark, handleRemark, user, CommonConstants.TWO_STRING);
			return R.ok("成功");
		} else {
			return R.failed("状态未变更，请刷新后查看列表状态！");
		}
	}

}
