/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TFascTemplate;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TFascTemplateDetail;
import com.yifu.cloud.plus.v1.yifu.archives.service.TFascTemplateDetailService;
import com.yifu.cloud.plus.v1.yifu.archives.service.TFascTemplateService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.TFascTemplateSearchVo;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 法大大专业版模板表
 *
 * @author hgw
 * @date 2025-09-30 16:29:28
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tfasctemplate")
@Tag(name = "法大大专业版模板表管理")
public class TFascTemplateController {

	private final TFascTemplateService tFascTemplateService;

	private final TFascTemplateDetailService tFascTemplateDetailService;


	/**
	 * 简单分页查询
	 *
	 * @param page          分页对象
	 * @param tFascTemplate 法大大专业版模板表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TFascTemplate>> getTFascTemplatePage(Page<TFascTemplate> page, TFascTemplateSearchVo tFascTemplate) {
		return new R<>(tFascTemplateService.getTFascTemplatePage(page, tFascTemplate));
	}

	/**
	 * 不分页查询
	 *
	 * @param tFascTemplate 法大大专业版模板表
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	public R<List<TFascTemplate>> getTFascTemplateNoPage(@RequestBody TFascTemplateSearchVo tFascTemplate) {
		return R.ok(tFascTemplateService.noPageDiy(tFascTemplate));
	}

	/**
	 * 通过id查询法大大专业版模板表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('archives_tfasctemplate_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('archives_tfasctemplate_get')")
	public R<TFascTemplate> getById(@PathVariable("id") String id) {
		return R.ok(tFascTemplateService.getById(id));
	}

	/**
	 * 修改法大大专业版模板表
	 *
	 * @param tFascTemplate 法大大专业版模板表
	 * @return R
	 */
	@Operation(summary = "修改法大大专业版模板表", description = "修改法大大专业版模板表：hasPermission('archives_tfasctemplate_edit')")
	@SysLog("修改法大大专业版模板表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('archives_tfasctemplate_edit')")
	public R<String> updateById(@RequestBody TFascTemplate tFascTemplate) {
		if (Common.isEmpty(tFascTemplate.getId())) {
			return R.failed("ID不能为空");
		}
		TFascTemplate old = tFascTemplateService.getById(tFascTemplate.getId());
		if (old == null) {
			return R.failed("未找到原模板");
		}
		if (Common.isEmpty(tFascTemplate.getContractType())) {
			return R.failed("合同类型不能为空");
		}
		if (Common.isNotNull(old.getContractType()) && old.getContractType().equals(tFascTemplate.getContractType())) {
			return R.failed("合同类型未变更");
		}
		old.setContractType(tFascTemplate.getContractType());
		tFascTemplateService.updateById(old);
		// 初始化字段，并重新计算模板的是否已配置
		tFascTemplateDetailService.initDetailHrField(tFascTemplate.getId());
		return R.ok();
	}


	/**
	 * 法大大专业版模板表 批量维护合同类型
	 *
	 * @author hgw
	 * @date 2025-09-30 16:29:28
	 **/
	@SneakyThrows
	@Operation(description = "批量维护合同类型 hasPermission('archives_tfasctemplate-batch-update')")
	@SysLog("批量维护合同类型")
	@PostMapping("/batchUpdate")
	public R<List<ErrorMessage>> batchUpdate(@RequestBody MultipartFile file) {
		return tFascTemplateService.batchUpdate(file.getInputStream());
	}

	/**
	 * 法大大专业版模板表 批量导出
	 *
	 * @author hgw
	 * @date 2025-09-30 16:29:28
	 **/
	@Operation(description = "导出法大大专业版模板表 hasPermission('archives_tfasctemplate-export')")
	@PostMapping("/export")
	public void export(HttpServletResponse response, @RequestBody TFascTemplateSearchVo searchVo) {
		tFascTemplateService.listExport(response, searchVo);
	}
}
