/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.pig4cloud.plugin.excel.vo.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysDataAuth;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeInfo;
import com.yifu.cloud.plus.v1.yifu.archives.service.TEmployeeInfoService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.DispatchCheckVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.DispatchEmpVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.EmpDispatchAddVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.ErrorMessageVO;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CacheConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.SecurityConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.config.DaprUpmsProperties;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.HttpDaprUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.cache.support.SimpleValueWrapper;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.util.List;
import java.util.Map;


/**
 * 人员档案表
 *
 * @author hgw
 * @date 2022-06-20 09:55:06
 */
@EnableConfigurationProperties({DaprUpmsProperties.class})
@RestController
@RequiredArgsConstructor
@RequestMapping("/temployeeinfo")
@Tag(name = "人员档案表管理")
public class TEmployeeInfoController {

	private final TEmployeeInfoService tEmployeeInfoService;

	// 缓存信息
	private final CacheManager cacheManager;

	private final DaprUpmsProperties daprUpmsProperties;

	/**
	 * 分页查询
	 *
	 * @param page          分页对象
	 * @param tEmployeeInfo 人员档案表
	 * @return
	 */
	@Operation(summary = "分页查询", description = "分页查询")
	@GetMapping("/page")
	public R<IPage<TEmployeeInfo>> getTEmployeeInfoPage(Page<TEmployeeInfo> page, TEmployeeInfo tEmployeeInfo) {
		YifuUser user = SecurityUtils.getUser();
		if (user == null || Common.isEmpty(user.getId())) {
			return R.failed(CommonConstants.PLEASE_LOG_IN);
		}
		// 普通用户：
		if (CommonConstants.ONE_STRING.equals(user.getSystemFlag())) {

			// 菜单id
			String menuId = "1536922631616278529";
			String linkId = user.getId();
			int linkType = 1; // 用户
			SysDataAuth sysDataAuth = new SysDataAuth();
			sysDataAuth.setLinkId(linkId);
			sysDataAuth.setLinkType(linkType);
			// 获取缓存菜单权限的步骤：
			Cache cache = cacheManager.getCache(CacheConstants.DATA_AUTH_DETAILS + linkType);
			Object obj = null;
			if (cache != null) {
				obj = cache.get(linkId + CommonConstants.DOWN_LINE_STRING + menuId);
				if (Common.isEmpty(obj)) {
					HttpDaprUtil.invokeMethodPost(daprUpmsProperties.getAppUrl(), daprUpmsProperties.getAppId()
							, "/dataAuth/refreshAuth", sysDataAuth, TEmployeeInfo.class, SecurityConstants.FROM_IN);
					obj = cache.get(linkId + CommonConstants.DOWN_LINE_STRING + menuId);
					if (Common.isEmpty(obj)) {
						linkId = user.getUserGroup();
						linkType = 0; // 用户组
						cache = cacheManager.getCache(CacheConstants.DATA_AUTH_DETAILS + linkType);
						if (cache != null) {
							obj = cache.get(linkId + CommonConstants.DOWN_LINE_STRING + menuId);
							if (Common.isEmpty(obj)) {
								sysDataAuth.setLinkId(linkId);
								sysDataAuth.setLinkType(linkType);
								HttpDaprUtil.invokeMethodPost(daprUpmsProperties.getAppUrl(), daprUpmsProperties.getAppId()
										, "/dataAuth/refreshAuth", sysDataAuth, TEmployeeInfo.class, SecurityConstants.FROM_IN);
								obj = cache.get(linkId + CommonConstants.DOWN_LINE_STRING + menuId);
							}
						}
					}
				}
			}
			if (Common.isNotNull(obj)) {
				SimpleValueWrapper objs = (SimpleValueWrapper) obj;
				if (objs != null) {
					String sql = String.valueOf(objs.get());
					if (sql.contains("#deptId")) {
						// TODO - 需要关联部门表
					}
					if (sql.contains("#create_by")) {
						sql = sql.replace("#create_by", user.getId());
					}
					if (sql.contains("#settleDomainId")) {
						// TODO - 获取人员结算主体权限
						// sql = sql.replace("#settleDomainId", ",'1','2' ")
						sql = sql.replace("or a.settle_domain_id in ('0'#settleDomainId) ", "");
					}
					if (sql.contains("#deptId")) {
						// TODO - 获取人员部门权限
						// sql = sql.replace("#settleDomainId", ",'1','2' ")
						sql = sql.replace(" or dept.dept_id = #deptId ", "");
					}
					return R.ok(tEmployeeInfoService.getPage(page, tEmployeeInfo, sql));
				}
			}
		}
		return R.ok(tEmployeeInfoService.getPage(page, tEmployeeInfo, null));
	}

	/**
	 * 分页查询
	 *
	 * @param page          分页对象
	 * @param tEmployeeInfo 人员档案表
	 * @return
	 */
	@Operation(summary = "离职库分页查询", description = "离职库分页查询")
	@GetMapping("/getLeavePage")
	public R<IPage<TEmployeeInfo>> getLeavePage(Page<TEmployeeInfo> page, TEmployeeInfo tEmployeeInfo) {
		YifuUser user = SecurityUtils.getUser();
		if (user == null || Common.isEmpty(user.getId())) {
			return R.failed(CommonConstants.PLEASE_LOG_IN);
		}
		// 普通用户：
		if (CommonConstants.ONE_STRING.equals(user.getSystemFlag())) {

			// 菜单id
			String menuId = "1536952884128591873";
			String linkId = user.getId();
			int linkType = 1; // 用户
			SysDataAuth sysDataAuth = new SysDataAuth();
			sysDataAuth.setLinkId(linkId);
			sysDataAuth.setLinkType(linkType);
			// 获取缓存菜单权限的步骤：
			Cache cache = cacheManager.getCache(CacheConstants.DATA_AUTH_DETAILS + linkType);
			Object obj = null;
			if (cache != null) {
				obj = cache.get(linkId + CommonConstants.DOWN_LINE_STRING + menuId);
				if (Common.isEmpty(obj)) {
					HttpDaprUtil.invokeMethodPost(daprUpmsProperties.getAppUrl(), daprUpmsProperties.getAppId()
							, "/dataAuth/refreshAuth", sysDataAuth, TEmployeeInfo.class, SecurityConstants.FROM_IN);
					obj = cache.get(linkId + CommonConstants.DOWN_LINE_STRING + menuId);
					if (Common.isEmpty(obj)) {
						linkId = user.getUserGroup();
						linkType = 0; // 用户组
						cache = cacheManager.getCache(CacheConstants.DATA_AUTH_DETAILS + linkType);
						if (cache != null) {
							obj = cache.get(linkId + CommonConstants.DOWN_LINE_STRING + menuId);
							if (Common.isEmpty(obj)) {
								sysDataAuth.setLinkId(linkId);
								sysDataAuth.setLinkType(linkType);
								HttpDaprUtil.invokeMethodPost(daprUpmsProperties.getAppUrl(), daprUpmsProperties.getAppId()
										, "/dataAuth/refreshAuth", sysDataAuth, TEmployeeInfo.class, SecurityConstants.FROM_IN);
								obj = cache.get(linkId + CommonConstants.DOWN_LINE_STRING + menuId);
							}
						}
					}
				}
			}
			if (Common.isNotNull(obj)) {
				SimpleValueWrapper objs = (SimpleValueWrapper) obj;
				if (objs != null) {
					String sql = String.valueOf(objs.get());
					if (sql.contains("#deptId")) {
						// TODO - 需要关联部门表
					}
					if (sql.contains("#create_by")) {
						sql = sql.replace("#create_by", user.getId());
					}
					if (sql.contains("#settleDomainId")) {
						// TODO - 获取人员结算主体权限
						// sql = sql.replace("#settleDomainId", ",'1','2' ")
						sql = sql.replace("or a.settle_domain_id in ('0'#settleDomainId) ", "");
					}
					if (sql.contains("#deptId")) {
						// TODO - 获取人员部门权限
						// sql = sql.replace("#settleDomainId", ",'1','2' ")
						sql = sql.replace(" or dept.dept_id = #deptId ", "");
					}
					return R.ok(tEmployeeInfoService.getLeavePage(page, tEmployeeInfo, sql));
				}
			}
		}
		return R.ok(tEmployeeInfoService.getLeavePage(page, tEmployeeInfo, null));
	}


	/**
	 * 通过id查询人员档案表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	public R<TEmployeeInfo> getById(@PathVariable("id") String id) {
		return R.ok(tEmployeeInfoService.getById(id));
	}

	/**
	 * @param empId
	 * @Description: 内部接口获取档案基本信息
	 * @Author: hgw
	 * @Date: 2022/7/21 14:47
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeInfo>
	 **/
	@Inner
	@PostMapping("/inner/getByEmpId")
	public TEmployeeInfo getByEmpId(@RequestBody String empId) {
		return tEmployeeInfoService.getById(empId);
	}

	/**
	 * @param empName   姓名
	 * @param empIdCard 身份证
	 * @Description: 新增人员档案前的校验
	 * @Author: hgw
	 * @Date: 2022/6/21 11:53
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.Boolean>
	 **/
	@Operation(summary = "新增人员档案前的校验", description = "新增人员档案前的校验")
	@SysLog("新增人员档案前的校验")
	@GetMapping("/checkIdCard")
	//@PreAuthorize("@pms.hasPermission('temployeeinfo_add')")
	public R<TEmployeeInfo> checkIdCard(@RequestParam String empName, @RequestParam String empIdCard) {
		TEmployeeInfo employeeInfo = new TEmployeeInfo();
		employeeInfo.setEmpName(empName);
		employeeInfo.setEmpIdcard(empIdCard);
		employeeInfo.setDeleteFlag(CommonConstants.STATUS_NORMAL);
		return tEmployeeInfoService.checkIdCard(employeeInfo);
	}

	/**
	 * 新增人员档案表
	 *
	 * @param tEmployeeInfo 人员档案表
	 * @return R
	 */
	@Operation(summary = "新增人员档案表（附带项目档案、学历）", description = "新增人员档案表")
	@SysLog("新增人员档案表（附带项目档案、学历）")
	@PostMapping("/addNewEmployee")
	//@PreAuthorize("@pms.hasPermission('temployeeinfo_add')")
	public R<String> addNewEmployee(@Valid @RequestBody TEmployeeInfo tEmployeeInfo) {
		return tEmployeeInfoService.addNewEmployee(tEmployeeInfo);
	}

	/**
	 * 修改人员档案表
	 *
	 * @param tEmployeeInfo 人员档案表
	 * @return R
	 */
	@Operation(summary = "修改人员档案表", description = "修改人员档案表")
	@SysLog("修改人员档案表")
	@PostMapping("/updateEmployee")
	@PreAuthorize("@pms.hasPermission('temployeeinfo_edit')")
	public R<Boolean> updateEmployee(@RequestBody TEmployeeInfo tEmployeeInfo) {
		return tEmployeeInfoService.updateEmployee(tEmployeeInfo);
	}

	/**
	 * @param id
	 * @Description: 通过id删除人员档案表
	 * @Author: hgw
	 * @Date: 2022/6/21 19:42
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "通过id删除人员档案表", description = "通过id删除人员档案表")
	@SysLog("通过id删除人员档案表")
	@GetMapping("/deleteEmployee")
	@PreAuthorize("@pms.hasPermission('temployeeinfo_del')")
	public R<String> deleteEmployee(@RequestParam String id) {
		return tEmployeeInfoService.deleteEmployee(id);
	}

	/**
	 * @param idList
	 * @Description: 通过ids删除人员档案表
	 * @Author: hgw
	 * @Date: 2022/6/21 19:42
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "通过idList批量删除人员档案表", description = "通过ids批量删除人员档案表")
	@SysLog("通过idList批量删除人员档案表")
	@PostMapping("/batchDeleteEmployee")
	@PreAuthorize("@pms.hasPermission('temployeeinfo_batch_del')")
	public R<List<ErrorMessageVO>> batchDeleteEmployee(@RequestBody List<String> idList) {
		return tEmployeeInfoService.batchDeleteEmployee(idList);
	}

	/**
	 * @param id
	 * @Description: 通过id减档人员档案表
	 * @Author: hgw
	 * @Date: 2022/6/21 19:42
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "通过id减档人员档案表temployeeinfo_minus", description = "通过id减档人员档案表")
	@SysLog("通过id减档人员档案表")
	@GetMapping("/minusEmployee")
	@PreAuthorize("@pms.hasPermission('temployeeinfo_minus')")
	public R<String> minusEmployee(@RequestParam String id, @RequestParam String leaveReason
			, @RequestParam(required = false) String leaveRemark) {
		return tEmployeeInfoService.minusEmployee(id, leaveReason, leaveRemark);
	}

	/**
	 * @Description: 通过ids批量减档人员档案表
	 * @Author: hgw
	 * @Date: 2022/6/21 19:42
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@SneakyThrows
	@Operation(summary = "批量减档人员档案表temployeeinfo_batch_minus", description = "批量减档人员档案表")
	@SysLog("批量减档人员档案表")
	@PostMapping("/batchMinusEmployee")
	@PreAuthorize("@pms.hasPermission('temployeeinfo_batch_minus')")
	public R<List<ErrorMessageVO>> batchMinusEmployee(@RequestBody MultipartFile file) {
		return tEmployeeInfoService.batchMinusEmployee(file.getInputStream());
	}

	/**
	 * @param id         离职档案id
	 * @param projectIds 需要复项的项目ids
	 * @Description: 档案复档
	 * @Author: hgw
	 * @Date: 2022/6/23 17:40
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "档案复档", description = "档案复档 projectIds：项目档案ids")
	@SysLog("档案复档")
	@GetMapping("/restoreEmployee")
	public R<String> restoreEmployee(@RequestParam String id, String projectIds) {
		return tEmployeeInfoService.restoreEmployee(id, projectIds);
	}

	/**
	 * @Description: 批量导入人员档案
	 * @Author: hgw
	 * @Date: 2022/6/22 11:57
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.pig4cloud.plugin.excel.vo.ErrorMessage>>
	 **/
	@SneakyThrows
	@Operation(summary = "批量导入人员档案", description = "批量导入人员档案 hasPermission('employee_batch_import') 额外属性：isCanAdd：0不可以复档复项；1允许复档复项")
	@SysLog("批量导入人员档案")
	@PostMapping("/batchImportEmployee")
	@PreAuthorize("@pms.hasPermission('employee_batch_import')")
	public R<List<ErrorMessage>> batchImportEmployee(@RequestBody MultipartFile file, @RequestParam Integer isCanAdd) {
		return tEmployeeInfoService.batchImportEmployee(file.getInputStream(), isCanAdd);
	}

	/**
	 * @Description: 批量更新人员档案
	 * @Author: hgw
	 * @Date: 2022/6/22 20:29
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.pig4cloud.plugin.excel.vo.ErrorMessage>>
	 **/
	@SneakyThrows
	@Operation(summary = "批量更新人员档案", description = "批量更新人员档案 hasPermission('employee_batch_update')")
	@SysLog("批量更新人员档案")
	@PostMapping("/batchUpdateEmployee")
	@PreAuthorize("@pms.hasPermission('employee_batch_update')")
	public R<List<ErrorMessage>> batchUpdateEmployee(@RequestBody MultipartFile file) {
		return tEmployeeInfoService.batchUpdateEmployee(file.getInputStream());
	}

	/**
	 * 批量导出人员档案信息
	 *
	 * @Author hgw
	 * @Date 2022-6-24 09:36:06
	 **/
	@Operation(summary = "批量导出人员档案信息", description = "批量导出人员档案信息 hasPermission('employee_export_employee')")
	@SysLog("批量导出人员档案信息")
	// @ResponseExcel
	@PostMapping("/exportEmployee")
	public void exportEmployee(@RequestBody(required = false) TEmployeeInfo employeeInfo, HttpServletResponse response) {
		tEmployeeInfoService.exportEmployee(employeeInfo, response);
	}

	/**
	 * 批量导出离职档案信息
	 *
	 * @Author hgw
	 * @Date 2022-6-24 09:36:06
	 **/
	@Operation(summary = "批量导出离职档案信息", description = "批量导出离职档案信息 hasPermission('employee_export_employee_leave')")
	@SysLog("批量导出离职档案信息")
	// @ResponseExcel
	@PostMapping("/exportLeaveEmployee")
	public void exportLeaveEmployee(@RequestBody(required = false) TEmployeeInfo employeeInfo, HttpServletResponse response) {
		tEmployeeInfoService.exportLeaveEmployee(employeeInfo, response);
	}

	/**
	 * 通过身份和编码查询档案 项目档案 合同信息
	 *
	 * @Author hgw
	 * @Date 2022-6-24 09:36:06
	 **/
	@Operation(description = "通过身份和编码查询档案 项目档案 合同信息")
	@SysLog("通过身份和编码查询档案 项目档案 合同信息")
	@Inner
	@PostMapping("/getDispatchEmpVo")
	public Map<String, DispatchEmpVo> getDispatchEmpVo(@RequestBody(required = true)DispatchCheckVo checkVo) {
		return  tEmployeeInfoService.getDispatchEmpVo(checkVo);
	}

	/**
	 * @Author fxj
	 * @Description 新增档案|项目档案|合同信息
	 * @Date 21:05 2022/7/20
	 * @Param
	 * @return
	**/
	@Operation(description = "新增档案|项目档案|合同信息")
	@SysLog("新增档案|项目档案|合同信息")
	@Inner
	@PostMapping("/addDispatchInfo")
	public EmpDispatchAddVo addDispatchInfo(@RequestBody(required = true) EmpDispatchAddVo addVo) {
		return  tEmployeeInfoService.addDispatchInfo(addVo);
	}

}
