/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TSettleDomain;
import com.yifu.cloud.plus.v1.yifu.archives.service.TSettleDomainService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.*;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;


/**
 * 结算主体表
 *
 * @author hgw
 * @date 2022-06-20 10:02:28
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tsettledomain")
@Tag(name = "结算主体表管理")
public class TSettleDomainController {

	private final TSettleDomainService tSettleDomainService;

	/**
	 * 分页查询
	 *
	 * @param page          分页对象
	 * @param tSettleDomain 结算主体表
	 * @return
	 */
	@Operation(summary = "分页查询", description = "分页查询")
	@GetMapping("/page")
	@PreAuthorize("@pms.hasPermission('demo_tsettledomain_get')")
	public R<IPage<TSettleDomain>> getTSettleDomainPage(Page<TSettleDomain> page, TSettleDomain tSettleDomain) {
		return R.ok(tSettleDomainService.page(page, Wrappers.query(tSettleDomain)));
	}


	/**
	 * 通过id查询结算主体表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('demo_tsettledomain_get')")
	public R<TSettleDomain> getById(@PathVariable("id") String id) {
		return R.ok(tSettleDomainService.getById(id));
	}

	/**
	 * 新增结算主体表
	 *
	 * @param tSettleDomain 结算主体表
	 * @return R
	 */
	@Operation(summary = "新增结算主体表", description = "新增结算主体表")
	@SysLog("新增结算主体表")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('demo_tsettledomain_add')")
	public R<Boolean> save(@RequestBody TSettleDomain tSettleDomain) {
		return R.ok(tSettleDomainService.save(tSettleDomain));
	}

	/**
	 * 修改结算主体表
	 *
	 * @param tSettleDomain 结算主体表
	 * @return R
	 */
	@Operation(summary = "修改结算主体表", description = "修改结算主体表")
	@SysLog("修改结算主体表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('demo_tsettledomain_edit')")
	public R<Boolean> updateById(@RequestBody TSettleDomain tSettleDomain) {
		return R.ok(tSettleDomainService.updateById(tSettleDomain));
	}

	/**
	 * 通过id删除结算主体表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除结算主体表", description = "通过id删除结算主体表")
	@SysLog("通过id删除结算主体表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('demo_tsettledomain_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tSettleDomainService.removeById(id));
	}

	/**
	 * 获取登录用户拥有的结算主体信息及单位部分信息
	 *
	 * @param
	 * @return
	 * @Author fxj
	 * @Date 2019-08-28
	 **/
	@Operation(description = "获取登录用户拥有的结算主体数据）")
	@Inner
	@GetMapping("/getSettleDomainIdsByUserId")
	public List<String> getSettleDomainIdsByUserId() {
		YifuUser user = SecurityUtils.getUser();
		if (null == user || null == user.getId()) {
			return null;
		}
		return tSettleDomainService.getSettleDomainIdsByUserId(user.getId());
	}

	/**
	 * @param
	 * @Author: wangan
	 * @Date: 2019/10/18
	 * @Description:通过id获取结算主体及单位部分信息
	 **/
	@Inner
	@PostMapping("/getSettleDomainSelectVoById")
	public R<TSettleDomainSelectVo> getInnerSettleDomainSelectVoByUserId(@RequestParam String id) {
		return new R<>(tSettleDomainService.selectSettleDomainSelectVosById(id));
	}

	/**
	 * @param
	 * @Author: wangan
	 * @Date: 2019/10/18
	 * @Description:获取所有客户单位的结算主体信息
	 **/
	@Inner
	@PostMapping("/selectAllSettleDomainSelectVos")
	public  R<List<TSettleDomainSelectVo>> selectAllSettleDomainSelectVos() {
		return new R<>(tSettleDomainService.selectAllSettleDomainSelectVos());
	}

	/**
	 * @Author fxj
	 * @Description  通过结算主体编码获取结算主体及客户信息
	 * @Date 10:49 2022/7/20
	 * @Param
	 * @return
	**/
	@Operation(description = "通过结算主体编码获取结算主体及客户信息")
	@SysLog("通过结算主体编码获取结算主体及客户信息")
	@Inner
	@PostMapping("/getSetInfoByCodes")
	public SetInfoVo getSetInfoByCodes(@RequestBody(required = true) List<String> codes) {
		return  tSettleDomainService.getSetInfoByCodes(codes);
	}
}
