/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.social.entity.SysBaseSetInfo;
import com.yifu.cloud.plus.v1.yifu.social.entity.SysPayProportion;
import com.yifu.cloud.plus.v1.yifu.social.service.SysBaseSetInfoService;
import com.yifu.cloud.plus.v1.yifu.social.service.SysPayProportionService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;


/**
 * 基数设置表
 *
 * @author hgw
 * @date 2022-07-11 18:21:23
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/sysbasesetinfo")
@Tag(name = "基数设置表管理")
public class SysBaseSetInfoController {

	private final SysBaseSetInfoService sysBaseSetInfoService;
	private final SysPayProportionService sysPayProportionService;


	/**
	 * 简单分页查询
	 *
	 * @param page           分页对象
	 * @param sysBaseSetInfo 基数设置表
	 * @return
	 */
	@Operation(summary = "分页查询", description = "分页查询")
	@GetMapping("/page")
	public R<IPage<SysBaseSetInfo>> getSysBaseSetInfoPage(Page<SysBaseSetInfo> page, SysBaseSetInfo sysBaseSetInfo) {
		return new R<>(sysBaseSetInfoService.getSysBaseSetInfoPage(page, sysBaseSetInfo));
	}

	/**
	 * 不分页查询
	 *
	 * @param sysBaseSetInfo 基数设置表
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	public R<List<SysBaseSetInfo>> getSysBaseSetInfoNoPage(@RequestBody SysBaseSetInfo sysBaseSetInfo) {
		return R.ok(sysBaseSetInfoService.list(Wrappers.query(sysBaseSetInfo).orderByDesc(CommonConstants.CREATE_TIME)));
	}

	/**
	 * 通过id查询基数设置表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	public R<SysBaseSetInfo> getById(@PathVariable("id") String id) {
		SysBaseSetInfo baseSetInfo = sysBaseSetInfoService.getById(id);
		if (null != baseSetInfo && CommonConstants.ONE_STRING.equals(baseSetInfo.getBaseType())){
			baseSetInfo.setFundProList(sysPayProportionService.list(Wrappers.<SysPayProportion>query().lambda().eq(SysPayProportion::getSysBaseSetId,id)));
		}
		return R.ok(baseSetInfo);
	}

	/**
	 * 新增基数设置表
	 *
	 * @param sysBaseSetInfo 基数设置表
	 * @return R
	 */
	@Operation(summary = "新增基数设置表", description = "新增基数设置表：hasPermission('social_sysbasesetinfo_add')")
	@SysLog("新增基数设置表")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('social_sysbasesetinfo_add')")
	public R<String> save(@RequestBody SysBaseSetInfo sysBaseSetInfo) {
		return sysBaseSetInfoService.saveSysBase(sysBaseSetInfo);
	}

	/**
	 * 修改基数设置表
	 *
	 * @param sysBaseSetInfo 基数设置表
	 * @return R
	 */
	@Operation(summary = "修改基数设置表", description = "修改基数设置表：hasPermission('social_sysbasesetinfo_edit')")
	@SysLog("修改基数设置表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('social_sysbasesetinfo_edit')")
	public R<String> updateById(@RequestBody SysBaseSetInfo sysBaseSetInfo) {
		return sysBaseSetInfoService.updateSysBase(sysBaseSetInfo);
	}

	/**
	 * 通过id删除基数设置表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除基数设置表", description = "通过id删除基数设置表：hasPermission('social_sysbasesetinfo_del')")
	@SysLog("通过id删除基数设置表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_sysbasesetinfo_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(sysBaseSetInfoService.removeById(id));
	}

	/**
	 * 通过id禁用启用一条记录
	 * @param id
	 * @return R
	 */
	@Operation(summary = "启用禁用", description = "启用禁用：hasPermission('social_sysbasesetinfo_enable') 0.在用 1.终止 2.过期")
	@SysLog("禁用启用基数设置表")
	@PostMapping("/disableOrEnableById")
	@PreAuthorize("@pms.hasPermission('wxhr:social_sysbasesetinfo_enable')")
	public R disableOrEnableById(@RequestParam String id, @RequestParam Integer status) {
		SysBaseSetInfo baseSetInfo = sysBaseSetInfoService.getById(id);
		if (null == baseSetInfo){
			return R.failed("无对应ID的数据！");
		}
		baseSetInfo.setStatus(status);
		return new R<>(sysBaseSetInfoService.updateById(baseSetInfo));
	}

}
