/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.social.constants.DispatchConstants;
import com.yifu.cloud.plus.v1.yifu.social.entity.TDispatchInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.TDispatchInfoService;
import com.yifu.cloud.plus.v1.yifu.social.vo.DispatchDetailVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.TDispatchInfoSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.log4j.Log4j2;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 派单信息记录表
 *
 * @author fxj
 * @date 2022-07-15 11:38:05
 */
@Log4j2
@RestController
@RequiredArgsConstructor
@RequestMapping("/tdispatchinfo" )
@Tag(name = "派单信息记录表管理")
public class TDispatchInfoController {

    private final TDispatchInfoService tDispatchInfoService;

    
    /**
     * 简单分页查询
     * @param page 分页对象
     * @param tDispatchInfo 派单信息记录表
     * @return
     */
    @Operation(description = "简单分页查询")
    @GetMapping("/page")
    public R<IPage<TDispatchInfo>> getTDispatchInfoPage(Page<TDispatchInfo> page, TDispatchInfoSearchVo tDispatchInfo) {
        return  new R<>(tDispatchInfoService.getTDispatchInfoPage(page,tDispatchInfo));
    }

    /**
     * 不分页查询
     * @param tDispatchInfo 派单信息记录表
     * @return
     */
    @Operation(summary = "不分页查询", description = "不分页查询")
    @PostMapping("/noPage" )
    //@PreAuthorize("@pms.hasPermission('demo_tdispatchinfo_get')" )
    public R<List<TDispatchInfo>> getTDispatchInfoNoPage(@RequestBody TDispatchInfoSearchVo tDispatchInfo) {
        return R.ok(tDispatchInfoService.noPageDiy(tDispatchInfo));
    }
	/**
	 * 简单分页查询--申请
	 * @param page 分页对象
	 * @param tDispatchInfo 派单信息记录表--申请
	 * @return
	 */
	@Operation(description = "简单分页查询--申请")
	@GetMapping("/pageApply")
	public R<IPage<TDispatchInfo>> getTDispatchApplyPage(Page<TDispatchInfo> page, TDispatchInfoSearchVo tDispatchInfo) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return  R.failed(CommonConstants.USER_FAIL);
		}
		tDispatchInfo.setCreateBy(user.getId());
		return  new R<>(tDispatchInfoService.getTDispatchInfoPage(page,tDispatchInfo));
	}

	/**
	 * 简单分页查询--审核
	 * @param page 分页对象
	 * @param tDispatchInfo 派单信息记录表--审核
	 * @return
	 */
	@Operation(description = "简单分页查询--审核")
	@GetMapping("/pageAudit")
	public R<IPage<TDispatchInfo>> getTDispatchAuditPage(Page<TDispatchInfo> page, TDispatchInfoSearchVo tDispatchInfo) {
		tDispatchInfo.setStatus(CommonConstants.ZERO_STRING);
		return  new R<>(tDispatchInfoService.getTDispatchInfoPage(page,tDispatchInfo));
	}

	/**
	 * 简单分页查询--审核
	 * @param page 分页对象
	 * @param tDispatchInfo 派单信息记录表--审核
	 * @return
	 */
	@Operation(description = "简单分页查询--审核")
	@GetMapping("/pageHandle")
	public R<IPage<TDispatchInfo>> getTDispatchHandlePage(Page<TDispatchInfo> page, TDispatchInfoSearchVo tDispatchInfo) {
		tDispatchInfo.setStatus(CommonConstants.ONE_STRING);
		return  new R<>(tDispatchInfoService.getTDispatchInfoPage(page,tDispatchInfo));
	}
    /**
     * 通过id查询派单信息记录表
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询：hasPermission('demo_tdispatchinfo_get')")
    @GetMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_tdispatchinfo_get')" )
    public R<TDispatchInfo> getById(@PathVariable("id" ) String id) {
        return R.ok(tDispatchInfoService.getById(id));
    }

	/**
	 * 通过id查询派单相关系信息
	 * @author fxj
	 * @param id id
	 * @return R
	 * @date 2022-07-25 10:19
	 */
	@Operation(summary = "通过id查询派单相关系信息", description = "通过id查询派单相关系信息")
	@GetMapping("/getSocialAndFundInfoById" )
	public R<DispatchDetailVo> getSocialAndFundInfoById(@RequestParam(value = "id") String id) {
		return R.ok(tDispatchInfoService.getSocialAndFundInfoById(id));
	}


	/**
     * 新增派单信息记录表
     * @param tDispatchInfo 派单信息记录表
     * @return R
     */
    @Operation(summary = "新增派单信息记录表", description = "新增派单信息记录表：hasPermission('demo_tdispatchinfo_add')")
    @SysLog("新增派单信息记录表" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('demo_tdispatchinfo_add')" )
    public R<Boolean> save(@RequestBody TDispatchInfo tDispatchInfo) {
        return R.ok(tDispatchInfoService.save(tDispatchInfo));
    }

    /**
     * 修改派单信息记录表
     * @param tDispatchInfo 派单信息记录表
     * @return R
     */
    @Operation(summary = "修改派单信息记录表", description = "修改派单信息记录表：hasPermission('demo_tdispatchinfo_edit')")
    @SysLog("修改派单信息记录表" )
    @PutMapping
    @PreAuthorize("@pms.hasPermission('demo_tdispatchinfo_edit')" )
    public R<Boolean> updateById(@RequestBody TDispatchInfo tDispatchInfo) {
        return R.ok(tDispatchInfoService.updateById(tDispatchInfo));
    }

    /**
     * 通过id删除派单信息记录表
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除派单信息记录表", description = "通过id删除派单信息记录表：hasPermission('demo_tdispatchinfo_del')")
    @SysLog("通过id删除派单信息记录表" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_tdispatchinfo_del')" )
    public R<Boolean> removeById(@PathVariable String id) {
        return tDispatchInfoService.removeByIdDiy(id);
    }

    /**
	 * 派单信息记录表 批量导入
	 *
     * @author fxj
     * @date 2022-07-15 11:38:05
	 **/
    @SneakyThrows
    @Operation(description = "批量新增派单信息记录表 hasPermission('demo_tdispatchinfo-batch-import')")
    @SysLog("批量新增派单信息记录表")
    @PostMapping("/importListAdd")
    //@PreAuthorize("@pms.hasPermission('demo_tdispatchinfo-batch-import')")
    public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file){
        return tDispatchInfoService.importDiy(file.getInputStream());
    }

    /**
     * 派单信息记录表 批量导出
     * @author fxj
     * @date 2022-07-15 11:38:05
     **/
    @Operation(description = "导出派单信息记录表 hasPermission('demo_tdispatchinfo-export')")
    @PostMapping("/export")
    @PreAuthorize("@pms.hasPermission('demo_tdispatchinfo-export')")
    public void export(HttpServletResponse response, @RequestBody TDispatchInfoSearchVo searchVo) {
            tDispatchInfoService.listExport(response,searchVo);
    }

	/**
	 * 派单信息记录表 批量导入-派减
	 *
	 * @author fxj
	 * @date 2022-07-15 11:38:05
	 **/
	@SneakyThrows
	@Operation(description = "派单信息记录表 批量导入-派减 hasPermission('demo_tdispatchinfo-batch-reduce')")
	@SysLog("派单信息记录表 批量导入-派减")
	@PostMapping("/importListReduce")
	//@PreAuthorize("@pms.hasPermission('demo_tdispatchinfo-batch-reduce')")
	public R<List<ErrorMessage>> importListReduce(@RequestBody MultipartFile file){
		return tDispatchInfoService.importReduceDiy(file.getInputStream());
	}

	/**
	 * @Author fxj
	 * @Description 派单申请批量审核
	 * @Date 16:54 2022/7/25
	 * @Param
	 * @return
	**/
	@Operation(description = "派单申请批量审核，传参派单ID hasPermission('wxhr:tdispatchinfo_addApplyAudit')")
	@PostMapping("/addApplyAudit")
	@PreAuthorize("@pms.hasPermission('wxhr:tdispatchinfo_addApplyAudit')")
	public R<List<ErrorMessage>> addApplyAudit(@RequestParam(name = "ids", required = true) String ids,
									@RequestParam(name = "auditStatus", required = true) String auditStatus,
									@RequestParam(name = "auditRemark", required = false) String auditRemark) {
		YifuUser user = SecurityUtils.getUser();
		if (null == user) {
			return R.failed(CommonConstants.USER_FAIL);
		}
		try {
			if (!Common.isNotNull(ids)) {
				return R.failed(DispatchConstants.DISPATCH_ID_NOT_EMPTY);
			}
			Integer flag = 1;
			if (CommonConstants.ONE_STRING.equals(auditStatus)) {
				//审核通过
				flag = CommonConstants.ZERO_INT;
			} else if (CommonConstants.TWO_STRING.equals(auditStatus)) {
				//审核不通过
				flag = CommonConstants.ONE_INT;
			} else {
				return R.failed("派单审核状态参数异常：0待审核  1审核通过 2审核不通过");
			}
			List<ErrorMessage> errorInfo = tDispatchInfoService.addBatchApplyAudit(Common.initStrToList(ids, CommonConstants.COMMA_STRING), user,  flag, auditStatus, null==auditRemark?"":auditRemark);
			return R.ok(errorInfo,"派单审核结果：");
		} catch (Exception e) {
			log.error("派单审核异常：" + e.getMessage());
			return R.failed("数据异常，派单审核失败!");
		}
	}
}
