/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.social.entity.TForecastLibrary;
import com.yifu.cloud.plus.v1.yifu.social.service.TForecastLibraryService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpHeaders;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 预估费用
 *
 * @author hgw
 * @date 2022-07-18 16:21:40
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tforecastlibrary")
@Tag(name = "预估费用管理")
@SecurityRequirement(name = HttpHeaders.AUTHORIZATION)
public class TForecastLibraryController {

	private final TForecastLibraryService tForecastLibraryService;


	/**
	 * 简单分页查询
	 *
	 * @param page             分页对象
	 * @param tForecastLibrary 预估费用
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TForecastLibrary>> getTForecastLibraryPage(Page<TForecastLibrary> page, TForecastLibrary tForecastLibrary) {
		return new R<>(tForecastLibraryService.getTForecastLibraryPage(page, tForecastLibrary));
	}

	/**
	 * 不分页查询
	 *
	 * @param tForecastLibrary 预估费用
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	//@PreAuthorize("@pms.hasPermission('social_tforecastlibrary_get')" )
	public R<List<TForecastLibrary>> getTForecastLibraryNoPage(@RequestBody TForecastLibrary tForecastLibrary) {
		return R.ok(tForecastLibraryService.noPageDiy(tForecastLibrary));
	}

	/**
	 * 通过id查询预估费用
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('social_tforecastlibrary_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tforecastlibrary_get')")
	public R<TForecastLibrary> getById(@PathVariable("id") String id) {
		return R.ok(tForecastLibraryService.getById(id));
	}

	/**
	 * 新增预估费用
	 *
	 * @param tForecastLibrary 预估费用
	 * @return R
	 */
	@Operation(summary = "新增预估费用", description = "新增预估费用：hasPermission('social_tforecastlibrary_add')")
	@SysLog("新增预估费用")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('social_tforecastlibrary_add')")
	public R<Boolean> save(@RequestBody TForecastLibrary tForecastLibrary) {
		return R.ok(tForecastLibraryService.save(tForecastLibrary));
	}

	/**
	 * 修改预估费用
	 *
	 * @param tForecastLibrary 预估费用
	 * @return R
	 */
	@Operation(summary = "修改预估费用", description = "修改预估费用：hasPermission('social_tforecastlibrary_edit')")
	@SysLog("修改预估费用")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('social_tforecastlibrary_edit')")
	public R<Boolean> updateById(@RequestBody TForecastLibrary tForecastLibrary) {
		return R.ok(tForecastLibraryService.updateById(tForecastLibrary));
	}

	/**
	 * 通过id删除预估费用
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除预估费用", description = "通过id删除预估费用：hasPermission('social_tforecastlibrary_del')")
	@SysLog("通过id删除预估费用")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tforecastlibrary_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tForecastLibraryService.removeById(id));
	}

	/**
	 * 预估费用 批量导出
	 *
	 * @author hgw
	 * @date 2022-07-18 16:21:40
	 **/
	@Operation(description = "导出预估费用 hasPermission('social_tforecastlibrary-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('social_tforecastlibrary-export')")
	public void export(HttpServletResponse response, @RequestBody TForecastLibrary searchVo) {
		tForecastLibraryService.listExport(response, searchVo);
	}

	/**
	 * @param payMonths 缴纳月
	 * @param empIdCard 身份证
	 * @param settleDomainIds 项目ids
	 * @Description: 按缴纳月重新生成
	 * @Author: hgw
	 * @Date: 2022/7/18 17:12
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.Boolean>
	 **/
	@Operation(summary = "按缴纳月重新生成", description = "按缴纳月重新生成：hasPermission('social_tforecastlibrary_create')")
	@SysLog("按缴纳月重新生成")
	@PostMapping("/createForecastlibary")
	@PreAuthorize("@pms.hasPermission('social_tforecastlibrary_create')")
	public R<String> createForecastlibary(@RequestParam String payMonths,
										   @RequestParam(value = "empIdCard", required = false) String empIdCard,
										   @RequestParam(value = "settleDomainIds", required = false) String settleDomainIds) {
		if (Common.isEmpty(empIdCard) && Common.isEmpty(settleDomainIds)) {
			return R.failed("参数有误：身份证或项目ID不可为空！");
		}
		return tForecastLibraryService.createForecastlibary(payMonths, empIdCard, settleDomainIds);
	}

}
