/*
 * Copyright (c) 2020 yifu4cloud Authors. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.yifu.cloud.plus.v1.yifu.admin.service.impl;

import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysDict;
import com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysDictItem;
import com.yifu.cloud.plus.v1.yifu.admin.mapper.SysDictItemMapper;
import com.yifu.cloud.plus.v1.yifu.admin.mapper.SysDictMapper;
import com.yifu.cloud.plus.v1.yifu.admin.service.SysDictService;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CacheConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.enums.DictTypeEnum;
import com.yifu.cloud.plus.v1.yifu.common.core.exception.ErrorCodes;
import com.yifu.cloud.plus.v1.yifu.common.core.util.MsgUtils;
import com.yifu.cloud.plus.v1.yifu.common.mybatis.base.BaseEntity;
import lombok.RequiredArgsConstructor;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

/**
 * 字典表
 *
 * @author lengleng
 * @date 2019/03/19
 */
@Service
@RequiredArgsConstructor
public class SysDictServiceImpl extends ServiceImpl<SysDictMapper, SysDict> implements SysDictService {

	private final SysDictItemMapper dictItemMapper;

	/**
	 * 根据ID 删除字典
	 * @param id 字典ID
	 * @return
	 */
	@Override
	@Transactional(rollbackFor = Exception.class)
	@CacheEvict(value = CacheConstants.DICT_DETAILS, allEntries = true)
	public void removeDict(Long id) {
		SysDict dict = this.getById(id);
		// 系统内置
		Assert.state(!DictTypeEnum.SYSTEM.getType().equals(dict.getSystemFlag()),
				MsgUtils.getMessage(ErrorCodes.SYS_DICT_DELETE_SYSTEM));
		baseMapper.deleteById(id);
		dictItemMapper.delete(Wrappers.<SysDictItem>lambdaQuery().eq(SysDictItem::getDictId, id));
	}

	/**
	 * 更新字典
	 * @param dict 字典
	 * @return
	 */
	@Override
	@CacheEvict(value = CacheConstants.DICT_DETAILS, key = "#dict.type")
	public void updateDict(SysDict dict) {
		SysDict sysDict = this.getById(dict.getId());
		// 系统内置
		Assert.state(!DictTypeEnum.SYSTEM.getType().equals(sysDict.getSystemFlag()),
				MsgUtils.getMessage(ErrorCodes.SYS_DICT_UPDATE_SYSTEM));
		this.updateById(dict);
	}

	@Override
	@CacheEvict(value = CacheConstants.DICT_DETAILS, allEntries = true)
	public void clearDictCache() {

	}

	/**
	 * @Author fxj
	 * @Description
	 * @Date 10:49 2022/6/14
	 * @Param sysDict
	 * @return
	**/
	@Override
	public IPage<SysDict> pageAsso(Page page, SysDict sysDict) {
		return this.page(page,buildQueryWrapper(sysDict));
	}
	/**
	 * @Author fxj
	 * @Description 获取字典类型及字典项值的父级字典项名称
	 * @Date 10:22 2022/6/21
	 **/
	@Override
	public String getParentDicLabelById(String type, String value) {
		return baseMapper.getParentDicLabelById(type,value);
	}
	/**
	 * @Author fxj
	 * @Description 获取type 对应的子字典项数目
	 * @Date 10:21 2022/6/21
	 **/
	@Override
	public int selectChildDictItemCount(String parentItemType, String parentId) {
		return baseMapper.selectChildDictItemCount(parentItemType,parentId);
	}

	/**
	 * 构建查询的 wrapper
	 * @param sysDict 查询条件
	 * @return LambdaQueryWrapper
	 */
	private LambdaQueryWrapper buildQueryWrapper(SysDict sysDict) {
		LambdaQueryWrapper<SysDict> wrapper = Wrappers.lambdaQuery();
		if (StrUtil.isNotBlank(sysDict.getSystemFlag())){
			wrapper.eq(SysDict::getSystemFlag,sysDict.getSystemFlag());
		}
		if (StrUtil.isNotBlank(sysDict.getType())){
			wrapper.likeRight(SysDict::getType,sysDict.getType());
		}

		if (StrUtil.isNotBlank(sysDict.getDescription())){
			wrapper.likeRight(SysDict::getDescription,sysDict.getDescription());
		}
		wrapper.orderByDesc(BaseEntity::getCreateTime);
		return wrapper;
	}
}
