/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.fadada.api.bean.rsp.sign.SignUrlRsp;
import com.fadada.api.exception.ApiException;
import com.yifu.cloud.plus.v1.yifu.archives.entity.FddContractAttachInfo;
import com.yifu.cloud.plus.v1.yifu.archives.entity.FddContractInfo;
import com.yifu.cloud.plus.v1.yifu.archives.service.FddContractAttachInfoService;
import com.yifu.cloud.plus.v1.yifu.archives.service.FddContractInfoService;
import com.yifu.cloud.plus.v1.yifu.archives.utils.FddUtil;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpHeaders;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;
import java.util.Map;


/**
 * 法大大合同与员工合同信息
 *
 * @author fxj
 * @date 2022-06-29 16:21:51
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/fddcontractinfo" )
@Tag(name = "法大大合同与员工合同信息管理")
@SecurityRequirement(name = HttpHeaders.AUTHORIZATION)
public class FddContractInfoController {

    private final FddContractInfoService fddContractInfoService;

	private final FddContractAttachInfoService fddContractAttachInfoService;

    /**
     * 分页查询
     * @param page 分页对象
     * @param fddContractInfo 法大大合同与员工合同信息
     * @return
     */
    @Operation(summary = "分页查询", description = "分页查询")
    @GetMapping("/page" )
    public R getFddContractInfoPage(Page page, FddContractInfo fddContractInfo) {
        return R.ok(fddContractInfoService.page(page, Wrappers.query(fddContractInfo)));
    }

    /**
     * 通过id查询法大大合同与员工合同信息
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询：hasPermission('demo_fddcontractinfo_get')")
    @GetMapping("/{id}" )
    public R getById(@PathVariable("id" ) String id) {
        return R.ok(fddContractInfoService.getById(id));
    }

    /**
     * 新增法大大合同与员工合同信息
     * @param fddContractInfo 法大大合同与员工合同信息
     * @return R
     */
    @Operation(summary = "新增法大大合同与员工合同信息", description = "新增法大大合同与员工合同信息：hasPermission('demo_fddcontractinfo_add')")
    @SysLog("新增法大大合同与员工合同信息" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('demo_fddcontractinfo_add')" )
    public R save(@RequestBody FddContractInfo fddContractInfo) {
        return R.ok(fddContractInfoService.save(fddContractInfo));
    }

    /**
     * 修改法大大合同与员工合同信息
     * @param fddContractInfo 法大大合同与员工合同信息
     * @return R
     */
    @Operation(summary = "修改法大大合同与员工合同信息", description = "修改法大大合同与员工合同信息：hasPermission('demo_fddcontractinfo_edit')")
    @SysLog("修改法大大合同与员工合同信息" )
    @PutMapping
    @PreAuthorize("@pms.hasPermission('demo_fddcontractinfo_edit')" )
    public R updateById(@RequestBody FddContractInfo fddContractInfo) {
        return R.ok(fddContractInfoService.updateById(fddContractInfo));
    }

    /**
     * 通过id删除法大大合同与员工合同信息
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除法大大合同与员工合同信息", description = "通过id删除法大大合同与员工合同信息：hasPermission('demo_fddcontractinfo_del')")
    @SysLog("通过id删除法大大合同与员工合同信息" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_fddcontractinfo_del')" )
    public R removeById(@PathVariable String id) {
        return R.ok(fddContractInfoService.removeById(id));
    }

	/**
	 * @param fddContractInfo
	 * @Author: huyc
	 * @Date: 2022/6/30
	 * @Description: 发送法大大电子合同定稿任务
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "发送法大大电子合同定稿任务", description = "发送法大大电子合同定稿任务")
	@SysLog("发送法大大电子合同定稿任务" )
	@PostMapping("/send")
	public R<String> send(FddContractInfo fddContractInfo) throws ApiException {
		FddContractAttachInfo fddContractAttachInfo = fddContractAttachInfoService.getById(fddContractInfo.getId());
		YifuUser user = SecurityUtils.getUser();
		return fddContractInfoService.saveEmpContract(fddContractAttachInfo,user);
	}

	/**
	 * @param fddContractInfo
	 * @Author: huyc
	 * @Date: 2022/6/30
	 * @Description:作废重签
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "作废重签", description = "作废重签")
	@PostMapping("/invalidAndsend")
	public R<String> invalidAndsend(FddContractInfo fddContractInfo) throws ApiException {
		return fddContractInfoService.invalidAndsend(fddContractInfo);
	}

	/**
	 * @param fddContractInfo
	 * @Author: huyc
	 * @Date: 2022/6/30
	 * @Description:公司获取在线定稿地址
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "公司获取在线定稿地址", description = "公司获取在线定稿地址")
	@PostMapping("/getCompanyFillFileUrl")
	public R<String> getCompanyFillFileUrl(FddContractInfo fddContractInfo) throws ApiException {
		return fddContractInfoService.getFillFileUrl(fddContractInfo, FddUtil.COMPANY);
	}


	/**
	 * @param fddContractInfo
	 * @Author: huyc
	 * @Date: 2022/6/30
	 * @Description: 员工获取在线定稿地址
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "员工获取在线定稿地址", description = "员工获取在线定稿地址")
	@PostMapping("/getPersonFillFileUrl")
	public R<String> getPersonFillFileUrl(FddContractInfo fddContractInfo) throws ApiException {
		return fddContractInfoService.getFillFileUrl(fddContractInfo, FddUtil.PERSON);
	}


	/**
	 * @param fddContractInfo
	 * @Author: huyc
	 * @Date: 2022/6/30
	 * @Description: 引用模板创建签署任务
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "引用模板创建签署任务", description = "引用模板创建签署任务")
	@PostMapping("/createTaskByDraftId")
	public R<String> createTaskByDraftId(FddContractInfo fddContractInfo) throws ApiException {
		//获取法大大电子合同
		FddContractInfo entity = fddContractInfoService.getById(fddContractInfo.getId());
		if (entity == null) {
			return R.failed("未查询到相关记录");
		}
		YifuUser user = SecurityUtils.getUser();
		FddContractAttachInfo fddContractAttachInfo = fddContractAttachInfoService.getOne(Wrappers.<FddContractAttachInfo>query().lambda()
				.eq(FddContractAttachInfo::getContractId,entity.getContractId()));
		if (!user.getId().equals(fddContractAttachInfo.getCreateBy())) {
			return R.failed("当前用户不是电子合同导入用户,不能操作");
		}
		R<String> returnR = fddContractInfoService.createTaskByDraftId(entity);
		if (returnR != null) {
			//记录返回信息
			if(returnR.getCode()== CommonConstants.SUCCESS){
				fddContractInfo.setRemark("成功");
			}else {
				fddContractInfo.setRemark(returnR.getMsg());
			}
		}
		fddContractInfoService.updateById(fddContractInfo);
		return returnR;
	}

	/**
	 * @param fddContractInfo
	 * @Author: huyc
	 * @Date: 2022/6/30
	 * @Description: 获取签署链接
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "获取签署链接", description = "获取签署链接")
	@PostMapping("/getSignUrl")
	public R<Map<String, List<SignUrlRsp>>> getSignUrl(FddContractInfo fddContractInfo) throws ApiException {
		return fddContractInfoService.getSignUrl(fddContractInfo);
	}

	/**
	 * @param fddContractInfo
	 * @Author: huyc
	 * @Date: 2022/6/30
	 * @Description: 获取签署文件预览地址
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "获取签署文件预览地址", description = "获取签署文件预览地址")
	@PostMapping("/getSignPreviewUrl")
	public R<String> getSignPreviewUrl(FddContractInfo fddContractInfo) throws ApiException {
		return fddContractInfoService.getSignPreviewUrl(fddContractInfo);
	}

	/**
	 * @param fddContractInfo
	 * @Author: huyc
	 * @Date: 2022/6/30
	 * @Description: 签署文件下载
	 **/
	@Operation(summary = "签署文件下载", description = "签署文件下载")
	@PostMapping("/getBySignFileId")
	public void getBySignFileId(FddContractInfo fddContractInfo, HttpServletResponse response) throws ApiException, IOException {
		fddContractInfoService.getBySignFileId(fddContractInfo, response);
	}

	/**
	 * @param fddContractInfo
	 * @Author: huyc
	 * @Date: 2022/6/30
	 * @Description: 签署文件转移为合同附件
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "签署文件转移为合同附件", description = "签署文件转移为合同附件")
	@PostMapping("/moveFileToEmpConcat")
	public R<String> moveFileToEmpConcat(FddContractInfo fddContractInfo, HttpServletResponse response) throws ApiException, IOException {
		return fddContractInfoService.moveFileToEmpConcat(fddContractInfo, response);
	}
}
