/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.MenuUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TPaymentBySalaryVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TPaymentVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.UpdateSocialFoundVo;
import com.yifu.cloud.plus.v1.yifu.social.entity.TForecastLibrary;
import com.yifu.cloud.plus.v1.yifu.social.service.TForecastLibraryService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 预估费用
 *
 * @author hgw
 * @date 2022-07-18 16:21:40
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tforecastlibrary")
@Tag(name = "预估费用管理")
public class TForecastLibraryController {

	private final TForecastLibraryService tForecastLibraryService;

	private final MenuUtil menuUtil;

	/**
	 * 简单分页查询
	 *
	 * @param page             分页对象
	 * @param tForecastLibrary 预估费用
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TForecastLibrary>> getTForecastLibraryPage(Page<TForecastLibrary> page, TForecastLibrary tForecastLibrary) {
		//Ekp1.9.11查询条件 社保生成月和公积金生成月必填其一
		if (Common.isEmpty(tForecastLibrary.getSocialCreateMonth()) && Common.isEmpty(tForecastLibrary.getProvidentCreateMonth())) {
			return null;
		}
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, tForecastLibrary);
		if (Common.isNotNull(tForecastLibrary.getAuthSql()) && tForecastLibrary.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
			tForecastLibrary.setAuthSql(tForecastLibrary.getAuthSql().replace(CommonConstants.A_DEPT_ID, "a.SETTLE_DOMAIN_ID"));
		}
		return new R<>(tForecastLibraryService.getTForecastLibraryPage(page, tForecastLibrary));
	}

	/**
	 * 通过id查询预估费用
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	public R<TForecastLibrary> getById(@PathVariable("id") String id) {
		return R.ok(tForecastLibraryService.getById(id));
	}

	/**
	 * 新增预估费用
	 *
	 * @param tForecastLibrary 预估费用
	 * @return R
	 */
	@Operation(summary = "新增预估费用", description = "新增预估费用：hasPermission('social_tforecastlibrary_add')")
	@SysLog("新增预估费用")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('social_tforecastlibrary_add')")
	public R<Boolean> save(@RequestBody TForecastLibrary tForecastLibrary) {
		return R.ok(tForecastLibraryService.save(tForecastLibrary));
	}

	/**
	 * 修改预估费用
	 *
	 * @param tForecastLibrary 预估费用
	 * @return R
	 */
	@Operation(summary = "修改预估费用", description = "修改预估费用：hasPermission('social_tforecastlibrary_edit')")
	@SysLog("修改预估费用")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('social_tforecastlibrary_edit')")
	public R<Boolean> updateById(@RequestBody TForecastLibrary tForecastLibrary) {
		return R.ok(tForecastLibraryService.updateById(tForecastLibrary));
	}

	/**
	 * 通过id删除预估费用
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除预估费用", description = "通过id删除预估费用：hasPermission('social_tforecastlibrary_del')")
	@SysLog("通过id删除预估费用")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tforecastlibrary_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tForecastLibraryService.removeById(id));
	}

	/**
	 * 预估费用 批量导出
	 *
	 * @author hgw
	 * @date 2022-07-18 16:21:40
	 **/
	@Operation(description = "导出预估费用 hasPermission('social_tforecastlibrary-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('social_tforecastlibrary-export')")
	public void export(HttpServletResponse response, @RequestBody TForecastLibrary searchVo) {
		//Ekp1.9.11查询条件 社保生成月和公积金生成月必填其一
		if (Common.isEmpty(searchVo.getSocialCreateMonth()) && Common.isEmpty(searchVo.getProvidentCreateMonth())) {
			return;
		}
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, searchVo);
		if (Common.isNotNull(searchVo.getAuthSql()) && searchVo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
			searchVo.setAuthSql(searchVo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "a.SETTLE_DOMAIN_ID"));
		}
		tForecastLibraryService.listExport(response, searchVo);
	}

	/**
	 * @param payMonths 缴纳月
	 * @param empIdCard 身份证
	 * @param settleDomainIds 项目ids
	 * @Description: 按缴纳月重新生成
	 * @Author: hgw
	 * @Date: 2022/7/18 17:12
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.Boolean>
	 **/
	@Operation(summary = "按缴纳月重新生成", description = "按缴纳月重新生成：hasPermission('social_tforecastlibrary_create')")
	@SysLog("按缴纳月重新生成")
	@PostMapping("/createForecastlibary")
	@PreAuthorize("@pms.hasPermission('social_tforecastlibrary_create')")
	public R<String> createForecastlibary(@RequestParam String payMonths
			, @RequestParam(value = "empIdCard", required = false) String empIdCard
			, @RequestParam(value = "settleDomainIds", required = false) String settleDomainIds) {
		if (Common.isEmpty(empIdCard) && Common.isEmpty(settleDomainIds)) {
			return R.failed("参数有误：身份证或项目ID不可为空！");
		}
		return tForecastLibraryService.createForecastlibary(payMonths, empIdCard, settleDomainIds);
	}

	/**
	 * @Description: 每日定时刷新社保公积金信息、预估数据，根据新增的户数据
	 * @Author: hgw
	 * @Date: 2022/7/27 19:34
	 * @return: void
	 **/
	@Operation(summary = "每日定时刷新社保公积金信息、预估数据，根据新增的户数据", description = "每日定时刷新社保公积金信息、预估数据，根据新增的户数据")
	@SysLog("每日定时刷新社保公积金信息、预估数据，根据新增的户数据")
	@Inner
	@PostMapping("/inner/updateForecastLibaryBySysBase")
	public void updateForecastLibaryBySysBase() {
		tForecastLibraryService.updateForecastLibaryBySysBase(null);
	}

	/**
	 * @Description: 每月定时生成下月预估库数据
	 * @Author: hgw
	 * @Date: 2022/7/27 19:34
	 * @return: void
	 **/
	@Operation(summary = "每月定时生成下月预估库数据", description = "每月定时生成下月预估库数据")
	@SysLog("每月定时生成下月预估库数据")
	@Inner
	@PostMapping("/inner/everyMonthCreateForecastLibary")
	public void everyMonthCreateForecastLibary() {
		tForecastLibraryService.everyMonthCreateForecastLibary();
	}

	/**
	 * @Description: 薪资获取预估库
	 * @Author: hgw
	 * @Date: 2022/8/10 18:06
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.salary.vo.TPaymentVo>
	 **/
	@Inner
	@PostMapping("/inner/selectTForecastBySalaryVo")
	public TPaymentVo selectTForecastBySalaryVo(@RequestBody TPaymentBySalaryVo tPaymentBySalaryVo) {
		return tForecastLibraryService.selectTForecastBySalaryVo(tPaymentBySalaryVo);
	}

	//hgw 以下4个方法，薪资变更缴费库预估库结算状态专用
	/**
	 * @param idList
	 * @Description: 社保变为结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:22
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updateForecastSocialStatusToSettle")
	public int updateForecastSocialStatusToSettle(@RequestBody List<String> idList) {
		return tForecastLibraryService.updateForecastSocialStatusToSettle(idList);
	}

	/**
	 * @param idList
	 * @Description: 社保变为未结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:23
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updateForecastSocialStatusToNoSettle")
	public int updateForecastSocialStatusToNoSettle(@RequestBody List<String> idList) {
		return tForecastLibraryService.updateForecastSocialStatusToNoSettle(idList);
	}

	/**
	 * @param idList
	 * @Description: 公积金变为结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:23
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updateForecastFundStatusToSettle")
	public int updateForecastFundStatusToSettle(@RequestBody List<String> idList) {
		return tForecastLibraryService.updateForecastFundStatusToSettle(idList);
	}

	/**
	 * @param idList
	 * @Description: 公积金变为未结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:23
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updateForecastFundStatusToNoSettle")
	public int updateForecastFundStatusToNoSettle(@RequestBody List<String> idList) {
		return tForecastLibraryService.updateForecastFundStatusToNoSettle(idList);
	}

	/**
	 * @param list
	 * @Description: 更新项目单位信息
	 * @Author: huyc
	 * @Date: 2022/8/25
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updateForSocialAndFound")
	public void updateForSocialAndFound(@RequestBody UpdateSocialFoundVo list) {
		tForecastLibraryService.updateForSocialAndFound(list);
	}

	/**
	 * @Description: 定时任务推送社保预估明细的数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "定时任务推送社保预估明细的数据", description = "定时任务推送社保预估明细的数据")
	@SysLog("定时任务推送社保预估明细的数据")
	@Inner
	@PostMapping("/inner/createForecastInfo")
	public void createForecastInfo() {
		tForecastLibraryService.createForecastInfo();
	}

	/**
	 * @Description: 定时任务推送社保预估明细的数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 **/
	@Operation(summary = "定时生成公积金预估明细的数据", description = "定时生成公积金预估明细的数据")
	@SysLog("定时生成公积金预估明细的数据")
	@Inner
	@PostMapping("/inner/createForecastFundInfo")
	public void createForecastFundInfo() {
		tForecastLibraryService.createForecastFundInfo();
	}
}
