/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.yifu.cloud.plus.v1.yifu.archives.config.GzConfig;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TGzOfferInfo;
import com.yifu.cloud.plus.v1.yifu.archives.service.TGzOfferInfoService;
import com.yifu.cloud.plus.v1.yifu.archives.utils.GZSign;
import com.yifu.cloud.plus.v1.yifu.archives.utils.ReturnGz;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.client.RestTemplate;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;

/**
 * 瓜子交互控制器
 *
 * @author hgw
 * @date 2025-6-23 09:39:27
 */
@Log4j2
@RestController
@RequiredArgsConstructor
@RequestMapping("/gz/core")
@Tag(name = "瓜子交互控制器")
public class TGzController {

	@Autowired
	private GzConfig gzConfig;

	private final TGzOfferInfoService tGzOfferInfoService;

	private RestTemplate restTemplate = new RestTemplate();

	/**
	 * @Description: 接收瓜子推送过来的Offer信息
	 * @Author: hgw
	 * @Date: 2025/6/23 15:27
	 * @return: com.yifu.cloud.plus.v1.yifu.archives.utils.ReturnGz<java.lang.String>
	 **/
	@PostMapping("/saveOfferInfo")
	public ReturnGz<String> saveOfferInfo(@RequestBody TGzOfferInfo gzOfferInfo
			, @RequestParam String expires, @RequestParam String signature, @RequestParam String requestId
			, @RequestParam String appkey, @RequestParam String nonce) {

		Map<String, Object> params = new HashMap<>();
		params.put("expires", expires);
		params.put("signature", signature);
		params.put("requestId", requestId);
		params.put("appkey", appkey);
		params.put("nonce", nonce);
		// 2. 计算期望的签名
		String expectedSignature = GZSign.getSignature(params, gzConfig.getAppsecret());
		// 3. 比较签名是否一致
		if (!params.get("signature").equals(expectedSignature)) {
			return ReturnGz.failed("签名验证失败");
		}
		// 5. 签名验证通过，处理业务逻辑
		return this.saveData(gzOfferInfo);
	}

	/**
	 * 处理推送的业务数据
	 *
	 * @return 处理结果
	 */
	private ReturnGz<String> saveData(TGzOfferInfo tGzOfferInfo) {
		// 这里实现你的业务逻辑
		// 例如：解析参数、保存数据、触发后续处理等
		tGzOfferInfo.setCreateBy("3");
		tGzOfferInfo.setCreateName("瓜子推送");
		tGzOfferInfo.setCreateTime(LocalDateTime.now());
		tGzOfferInfoService.save(tGzOfferInfo);
		return ReturnGz.ok();
	}

	/**
	 * @Description: 获取瓜子银行支行信息的接口
	 * @Author: hgw
	 * @Date: 2025/6/23 15:28
	 * @return: boolean
	 **/
	@GetMapping("/getGzBank")
	public boolean getGzBank(@RequestParam String pageNum, @RequestParam String pageSize
			, @RequestParam String startTime, @RequestParam String endTime) {
		Map<String, Object> params = new HashMap<>();
		params.put("pageNum", pageNum);
		params.put("pageSize", pageSize);
		params.put("startTime", startTime);
		params.put("endTime", endTime);
		GZSign.addSignature(params, gzConfig.getAppkey(), gzConfig.getAppsecret());

		params.remove("startTime", startTime);
		params.remove("endTime", endTime);
		String sortedParams = GZSign.getEncodeString(params);
		sortedParams += "&startTime=" + startTime + "&endTime=" + endTime;
		return gzConfig.getGzBankInfo(restTemplate, sortedParams);
	}


}
