/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.salary.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TStatisticsBonus;
import com.yifu.cloud.plus.v1.yifu.salary.service.TStatisticsBonusService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TStatisticsBonusImportVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TStatisticsBonusSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 全年一次性奖金申报表
 *
 * @author huyc
 * @date 2022-08-14 21:31:15
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tstatisticsbonus" )
@Tag(name = "全年一次性奖金申报表管理")
public class TStatisticsBonusController {

    private final  TStatisticsBonusService tStatisticsBonusService;

    /**
     * 简单分页查询
     * @param page 分页对象
     * @param tStatisticsBonus 全年一次性奖金申报表
     * @return
     */
    @Operation(description = "简单分页查询")
    @GetMapping("/page")
    public R<IPage<TStatisticsBonus>> getTStatisticsBonusPage(Page<TStatisticsBonus> page, TStatisticsBonusSearchVo tStatisticsBonus) {
        return  new R<>(tStatisticsBonusService.getTStatisticsBonusPage(page,tStatisticsBonus));
    }
    /**
     * 不分页查询
     * @param tStatisticsBonus 全年一次性奖金申报表
     * @return
     */
    @Operation(summary = "不分页查询", description = "不分页查询")
    @PostMapping("/noPage" )
    public R<List<TStatisticsBonus>> getTStatisticsBonusNoPage(@RequestBody TStatisticsBonusSearchVo tStatisticsBonus) {
        return R.ok(tStatisticsBonusService.noPageDiy(tStatisticsBonus));
    }

    /**
     * 新增全年一次性奖金申报表
     * @param tStatisticsBonus 全年一次性奖金申报表
     * @return R
     */
    @Operation(summary = "新增全年一次性奖金申报表", description = "新增全年一次性奖金申报表：hasPermission('salary_tstatisticsbonus_add')")
    @SysLog("新增全年一次性奖金申报表" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('salary_tstatisticsbonus_add')" )
    public R<Boolean> save(@RequestBody TStatisticsBonus tStatisticsBonus) {
        return R.ok(tStatisticsBonusService.save(tStatisticsBonus));
    }

    /**
     * 通过id删除全年一次性奖金申报表
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除全年一次性奖金申报表", description = "通过id删除全年一次性奖金申报表：hasPermission('salary_tstatisticsbonus_del')")
    @SysLog("通过id删除全年一次性奖金申报表" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('salary_tstatisticsbonus_del')" )
    public R<Boolean> removeById(@PathVariable String id) {
        return R.ok(tStatisticsBonusService.removeById(id));
    }

    /**
	 * 全年一次性奖金测算
     * @author huyc
     * @date 2022-08-14 21:31:15
	 **/
    @SneakyThrows
    @Operation(description = "全年一次性奖金测算 hasPermission('salary_tstatisticsbonus-batch-import')")
    @SysLog("全年一次性奖金测算")
    @PostMapping("/importListAdd")
    @PreAuthorize("@pms.hasPermission('salary_tstatisticsbonus-batch-import')")
    public void importListAdd(HttpServletResponse response, @RequestBody List<TStatisticsBonusImportVo> list, String deptId){
        tStatisticsBonusService.importDiy(response,list,deptId);
    }

    /**
     * 全年一次性奖金申报表 批量导出
     * @author huyc
     * @date 2022-08-14 21:31:15
     **/
    @Operation(description = "导出全年一次性奖金申报表 hasPermission('salary_tstatisticsbonus-export')")
    @PostMapping("/export")
    @PreAuthorize("@pms.hasPermission('salary_tstatisticsbonus-export')")
    public void export(HttpServletResponse response, @RequestBody TStatisticsBonusSearchVo searchVo) {
		tStatisticsBonusService.listExport(response,searchVo);
    }

	/**
	 * 全年一次性奖金申报表 批量删除
	 *
	 * @author huyc
	 * @date 2022-08-14 21:31:15
	 **/
	@Operation(description = "批量删除全年一次性奖金申报表 hasPermission('salary_tstatisticsbonus_del')")
	@SysLog("批量删除全年一次性奖金申报表")
	@PostMapping("/batchDelete")
	@PreAuthorize("@pms.hasPermission('salary_tstatisticsbonus_del')")
	public R batchDelete(@RequestParam String declareMonth){
		return tStatisticsBonusService.batchDelete(declareMonth);
	}

	/**
	 * @param
	 * @Description: 生成全年一次性奖金
	 * @Author: huyc
	 * @Date: 2022-08-17 18:17:48
	 * @return: com.yifu.cloud.v1.common.core.util.R
	 **/
	@Inner
	@Operation(description = "生成全年一次性奖金")
	@PostMapping("/inner/doStatisticsBonus")
	public R doStatisticsBonus() {
		return tStatisticsBonusService.doStatisticsBonus();
	}
}
