/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.salary.service.impl;

import cn.hutool.core.bean.BeanUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.read.listener.ReadListener;
import com.alibaba.excel.read.metadata.holder.ReadRowHolder;
import com.alibaba.excel.util.ListUtils;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.*;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TStatisticsDeclarer;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TStatisticsLabor;
import com.yifu.cloud.plus.v1.yifu.salary.mapper.TStatisticsDeclarerMapper;
import com.yifu.cloud.plus.v1.yifu.salary.mapper.TStatisticsLaborMapper;
import com.yifu.cloud.plus.v1.yifu.salary.service.TStatisticsLaborService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TStatisticsLaborSearchVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TStatisticsLaborVo;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 本期劳务费申报表
 *
 * @author hgw
 * @date 2022-08-05 11:40:14
 */
@Log4j2
@Service
@RequiredArgsConstructor
public class TStatisticsLaborServiceImpl extends ServiceImpl<TStatisticsLaborMapper, TStatisticsLabor> implements TStatisticsLaborService {

	private final TStatisticsDeclarerMapper declarerMapper;

	/**
	 * 本期劳务费申报表简单分页查询
	 *
	 * @param tStatisticsLabor 本期劳务费申报表
	 * @return
	 */
	@Override
	public IPage<TStatisticsLabor> getTStatisticsLaborPage(Page<TStatisticsLabor> page, TStatisticsLaborSearchVo tStatisticsLabor) {
		return baseMapper.getTStatisticsLaborPage(page, tStatisticsLabor);
	}

	/**
	 * 本期劳务费申报表批量导出
	 *
	 * @return
	 */
	@Override
	public void listExport(HttpServletResponse response, TStatisticsLaborSearchVo searchVo) {
		String fileName = "本期劳务费导出" + DateUtil.getThisTime() + CommonConstants.XLSX;
		//获取要导出的列表
		List<TStatisticsLabor> list = new ArrayList<>();
		long count = noPageCountDiy(searchVo);
		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName, "UTF-8"));
			// 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
			ExcelWriter excelWriter = EasyExcel.write(out, TStatisticsLabor.class).build();
			int index = 0;
			if (count > CommonConstants.ZERO_INT) {
				WriteSheet writeSheet;
				ExcelUtil<TStatisticsLabor> util;
				for (int i = 0; i <= count; ) {
					// 获取实际记录
					searchVo.setLimitStart(i);
					searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
					list = noPageDiy(searchVo);
					if (Common.isNotNull(list)) {
						util = new ExcelUtil<>(TStatisticsLabor.class);
						for (TStatisticsLabor vo : list) {
							util.convertEntity(vo, null, null, null);
						}
					}
					if (Common.isNotNull(list)) {
						writeSheet = EasyExcel.writerSheet("本期劳务费申报表" + index).build();
						excelWriter.write(list, writeSheet);
						index++;
					}
					i = i + CommonConstants.EXCEL_EXPORT_LIMIT;
					if (Common.isNotNull(list)) {
						list.clear();
					}
				}
			} else {
				WriteSheet writeSheet = EasyExcel.writerSheet("本期劳务费申报表" + index).build();
				excelWriter.write(list, writeSheet);
			}
			if (Common.isNotNull(list)) {
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		} catch (Exception e) {
			log.error("执行异常", e);
		} finally {
			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	@Override
	public List<TStatisticsLabor> noPageDiy(TStatisticsLaborSearchVo searchVo) {
		LambdaQueryWrapper<TStatisticsLabor> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TStatisticsLabor::getId, idList);
		}
		if (searchVo.getLimitStart() >= 0 && searchVo.getLimitEnd() > 0) {
			wrapper.last(" limit " + searchVo.getLimitStart() + "," + searchVo.getLimitEnd());
		}
		return baseMapper.selectList(wrapper);
	}

	private Long noPageCountDiy(TStatisticsLaborSearchVo searchVo) {
		LambdaQueryWrapper<TStatisticsLabor> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TStatisticsLabor::getId, idList);
		}
		return baseMapper.selectCount(wrapper);
	}

	private LambdaQueryWrapper buildQueryWrapper(TStatisticsLaborSearchVo entity) {
		LambdaQueryWrapper<TStatisticsLabor> wrapper = Wrappers.lambdaQuery();
		if (Common.isNotNull(entity.getDeclareMonth())) {
			wrapper.eq(TStatisticsLabor::getDeclareMonth,entity.getDeclareMonth());
		}
		if (Common.isNotNull(entity.getEmpName())) {
			wrapper.eq(TStatisticsLabor::getEmpName,entity.getEmpName());
		}
		if (Common.isNotNull(entity.getEmpIdcard())) {
			wrapper.eq(TStatisticsLabor::getEmpIdcard,entity.getEmpIdcard());
		}
		return wrapper;
	}

	@Override
	public R<List<ErrorMessage>> importDiy(InputStream inputStream) {
		List<ErrorMessage> errorMessageList = new ArrayList<>();
		ExcelUtil<TStatisticsLaborVo> util1 = new ExcelUtil<>(TStatisticsLaborVo.class);
		;
		// 写法2：
		// 匿名内部类 不用额外写一个DemoDataListener
		// 这里 需要指定读用哪个class去读，然后读取第一个sheet 文件流会自动关闭
		try {
			EasyExcel.read(inputStream, TStatisticsLaborVo.class, new ReadListener<TStatisticsLaborVo>() {
				/**
				 * 单次缓存的数据量
				 */
				public static final int BATCH_COUNT = CommonConstants.BATCH_COUNT;
				/**
				 *临时存储
				 */
				private List<TStatisticsLaborVo> cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);

				@Override
				public void invoke(TStatisticsLaborVo data, AnalysisContext context) {
					ReadRowHolder readRowHolder = context.readRowHolder();
					Integer rowIndex = readRowHolder.getRowIndex();
					data.setRowIndex(rowIndex + 1);
					ErrorMessage errorMessage = util1.checkEntity(data, data.getRowIndex());
					if (Common.isNotNull(errorMessage)) {
						errorMessageList.add(errorMessage);
					} else {
						cachedDataList.add(data);
					}
					if (cachedDataList.size() >= BATCH_COUNT) {
						saveData();
						// 存储完成清理 list
						cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);
					}
				}

				@Override
				public void doAfterAllAnalysed(AnalysisContext context) {
					saveData();
				}

				/**
				 * 加上存储数据库
				 */
				private void saveData() {
					log.info("{}条数据，开始存储数据库！", cachedDataList.size());
					importTStatisticsLabor(cachedDataList, errorMessageList);
					log.info("存储数据库成功！");
				}
			}).sheet().doRead();
		} catch (Exception e) {
			log.error(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR, e);
			return R.failed(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR);
		}
		return R.ok(errorMessageList);
	}

	@Override
	public R batchDelete(String declareMonth) {
		if (Common.isNotNull(declareMonth)) {
			List<TStatisticsLabor> deleteList = baseMapper.selectList(Wrappers.<TStatisticsLabor>query().lambda()
					.eq(TStatisticsLabor::getDeclareMonth, declareMonth));
			if (Common.isNotNull(deleteList)) {
				List<String> idList = deleteList.stream().map(TStatisticsLabor::getId).collect(Collectors.toList());
				baseMapper.deleteBatchIds(idList);
				return R.ok();
			}
			return R.failed(CommonConstants.NO_DATA_TO_HANDLE);
		}
		return null;
	}

	@Override
	public R doStatisticsLabor() {
		String nowMonth = DateUtil.addMonth(0);  //本月
		TStatisticsLabor std = new TStatisticsLabor();
		std.setDeclareMonth(nowMonth);
		QueryWrapper<TStatisticsLabor> queryWrapperSs = new QueryWrapper<>();
		queryWrapperSs.setEntity(std);
		long num = this.count(queryWrapperSs);
		if (num > 0) {
			return R.failed("上月数据已生成，不可重复生成！");
		} else {
			String lastMonth =  DateUtil.addMonth(-1);  //上月
			List<TStatisticsLabor> stdvoList = baseMapper.doStatisticsLabor(nowMonth,lastMonth);
			for (TStatisticsLabor declarer : stdvoList) {
				this.save(declarer);
			}
			return R.ok();
		}
	}

	@Override
	public R refreshStatisticsLabor(String declareMonth) {
		TStatisticsLabor std = new TStatisticsLabor();
		std.setDeclareMonth(declareMonth);
		QueryWrapper<TStatisticsLabor> queryWrapperSs = new QueryWrapper<>();
		queryWrapperSs.setEntity(std);
		long num = this.count(queryWrapperSs);
		if (num > 0) {
			baseMapper.delete(Wrappers.<TStatisticsLabor>query().lambda().eq(TStatisticsLabor::getDeclareMonth,declareMonth));
		}
		String lastMonth =  DateUtil.addMonthByYearMonth(-1,declareMonth);  //上月
		List<TStatisticsLabor> stdvoList = baseMapper.doStatisticsLabor(declareMonth,lastMonth);
		for (TStatisticsLabor labor : stdvoList) {
			TStatisticsDeclarer declarer = declarerMapper.selectOne(Wrappers.<TStatisticsDeclarer>query().lambda()
					.eq(TStatisticsDeclarer::getDeclareMonth,declareMonth)
					.eq(TStatisticsDeclarer::getEmpIdcard,labor.getEmpIdcard())
					.eq(TStatisticsDeclarer::getDeclareUnit,labor.getDeclareUnit())
					.last(CommonConstants.LAST_ONE_SQL));
			if (Common.isNotNull(declarer) && CommonConstants.ZERO_STRING.equals(declarer.getIsDeclare())) {
				this.save(labor);
			}
		}
		return R.ok();
	}

	private void importTStatisticsLabor(List<TStatisticsLaborVo> excelVOList, List<ErrorMessage> errorMessageList) {
		// 个性化校验逻辑
		ErrorMessage errorMsg;
		// 执行数据插入操作 组装
		for (int i = 0; i < excelVOList.size(); i++) {
			TStatisticsLaborVo excel = excelVOList.get(i);

			// 插入
			insertExcel(excel);
			errorMessageList.add(new ErrorMessage(excel.getRowIndex(), CommonConstants.SAVE_SUCCESS));
		}
	}

	/**
	 * 插入excel bad record
	 */
	private void insertExcel(TStatisticsLaborVo excel) {
		TStatisticsLabor insert = new TStatisticsLabor();
		BeanUtil.copyProperties(excel, insert);
		this.save(insert);
	}
}
