/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.salary.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ArrayUtil;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.read.listener.ReadListener;
import com.alibaba.excel.read.metadata.holder.ReadRowHolder;
import com.alibaba.excel.util.ListUtils;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.*;
import com.yifu.cloud.plus.v1.yifu.common.mybatis.base.BaseEntity;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TBankSet;
import com.yifu.cloud.plus.v1.yifu.salary.mapper.TBankSetMapper;
import com.yifu.cloud.plus.v1.yifu.salary.service.TBankSetService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TBankSetSearchVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TBankSetVo;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 开户行配置表
 *
 * @author hgw
 * @date 2023-03-06 11:54:21
 */
@Log4j2
@Service
public class TBankSetServiceImpl extends ServiceImpl<TBankSetMapper, TBankSet> implements TBankSetService {
	/**
	 * 开户行配置表简单分页查询
	 *
	 * @param tBankSet 开户行配置表
	 * @return
	 */
	@Override
	public IPage<TBankSet> getTBankSetPage(Page<TBankSet> page, TBankSetSearchVo tBankSet) {
		return baseMapper.getTBankSetPage(page, tBankSet);
	}

	/**
	 * 开户行配置表简单分页查询
	 *
	 * @param tBankSet 开户行配置表
	 * @return
	 */
	@Override
	public List<TBankSet> getTBankSetIdNameList(TBankSetSearchVo tBankSet) {
		return baseMapper.getTBankSetIdNameList(tBankSet);
	}

	/**
	 * 开户行Map
	 * @param bankName 开户行
	 * @return
	 */
	@Override
	public Map<String, String> getBankMap(String bankName) {
		List<String> sList = baseMapper.getTBankSetList(bankName);
		Map<String, String> bankMap = new HashMap<>();
		if (sList != null && !sList.isEmpty()) {
			for (String s : sList) {
				bankMap.put(s,s);
			}
		}
		return bankMap;
	}

	/**
	 * 开户行配置表批量导出
	 *
	 * @param searchVo 开户行配置表
	 * @return
	 */
	@Override
	public void listExport(HttpServletResponse response, TBankSetSearchVo searchVo) {
		String fileName = "开户行配置表批量导出" + DateUtil.getThisTime() + ".xlsx";
		//获取要导出的列表
		List<TBankSet> list = new ArrayList<>();
		long count = noPageCountDiy(searchVo);
		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName, CommonConstants.UTF8));
			// 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
			ExcelWriter excelWriter = EasyExcelFactory.write(out, TBankSet.class).build();
			int index = 0;
			if (count > CommonConstants.ZERO_INT) {
				for (int i = 0; i <= count; i = i + CommonConstants.EXCEL_EXPORT_LIMIT) {
					// 获取实际记录
					searchVo.setLimitStart(i);
					searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
					list = noPageDiy(searchVo);
					if (Common.isNotNull(list)) {
						WriteSheet writeSheet = EasyExcelFactory.writerSheet("开户行配置表" + index).build();
						excelWriter.write(list, writeSheet);
						index++;
						list.clear();
					}
				}
			} else {
				WriteSheet writeSheet = EasyExcelFactory.writerSheet("开户行配置表" + index).build();
				excelWriter.write(list, writeSheet);
			}
			if (Common.isNotNull(list)) {
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		} catch (Exception e) {
			log.error("执行异常", e);
		} finally {
			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	@Override
	public List<TBankSet> noPageDiy(TBankSetSearchVo searchVo) {
		LambdaQueryWrapper<TBankSet> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TBankSet::getId, idList);
		}
		if (searchVo.getLimitStart() >= 0 && searchVo.getLimitEnd() > 0) {
			wrapper.last(" limit " + searchVo.getLimitStart() + "," + searchVo.getLimitEnd());
		}
		wrapper.orderByDesc(BaseEntity::getCreateTime);
		return baseMapper.selectList(wrapper);
	}

	private Long noPageCountDiy(TBankSetSearchVo searchVo) {
		LambdaQueryWrapper<TBankSet> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TBankSet::getId, idList);
		}
		return baseMapper.selectCount(wrapper);
	}

	private LambdaQueryWrapper buildQueryWrapper(TBankSetSearchVo entity) {
		LambdaQueryWrapper<TBankSet> wrapper = Wrappers.lambdaQuery();
		if (ArrayUtil.isNotEmpty(entity.getCreateTimes())) {
			wrapper.ge(TBankSet::getCreateTime, entity.getCreateTimes()[0])
					.le(TBankSet::getCreateTime,
							entity.getCreateTimes()[1]);
		}
		if (Common.isNotNull(entity.getCreateName())) {
			wrapper.eq(TBankSet::getCreateName, entity.getCreateName());
		}
		wrapper.eq(TBankSet:: getDeleteFlag, CommonConstants.STATUS_NORMAL);
		return wrapper;
	}

	@Override
	public R<List<ErrorMessage>> importDiy(InputStream inputStream) {
		List<ErrorMessage> errorMessageList = new ArrayList<>();
		ExcelUtil<TBankSetVo> util1 = new ExcelUtil<>(TBankSetVo.class);
		// 写法2：
		// 匿名内部类 不用额外写一个DemoDataListener
		// 这里 需要指定读用哪个class去读，然后读取第一个sheet 文件流会自动关闭
		try {
			EasyExcelFactory.read(inputStream, TBankSetVo.class, new ReadListener<TBankSetVo>() {
				/**
				 * 单次缓存的数据量
				 */
				public static final int BATCH_COUNT = CommonConstants.BATCH_COUNT;
				/**
				 *临时存储
				 */
				private List<TBankSetVo> cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);

				@Override
				public void invoke(TBankSetVo data, AnalysisContext context) {
					ReadRowHolder readRowHolder = context.readRowHolder();
					Integer rowIndex = readRowHolder.getRowIndex();
					data.setRowIndex(rowIndex + 1);
					ErrorMessage errorMessage = util1.checkEntity(data, data.getRowIndex());
					if (Common.isNotNull(errorMessage)) {
						errorMessageList.add(errorMessage);
					} else {
						cachedDataList.add(data);
					}
					if (cachedDataList.size() >= BATCH_COUNT) {
						saveData();
						// 存储完成清理 list
						cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);
					}
				}

				@Override
				public void doAfterAllAnalysed(AnalysisContext context) {
					saveData();
				}

				/**
				 * 加上存储数据库
				 */
				private void saveData() {
					log.info("{}条数据，开始存储数据库！", cachedDataList.size());
					importTBankSet(cachedDataList, errorMessageList);
					log.info("存储数据库成功！");
				}
			}).sheet().doRead();
		} catch (Exception e) {
			log.error(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR, e);
			return R.failed(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR);
		}
		return R.ok(errorMessageList);
	}

	private void importTBankSet(List<TBankSetVo> excelVOList, List<ErrorMessage> errorMessageList) {
		// 执行数据插入操作 组装
		for (int i = 0; i < excelVOList.size(); i++) {
			TBankSetVo excel = excelVOList.get(i);
			// 插入
			insertExcel(excel);
			errorMessageList.add(new ErrorMessage(excel.getRowIndex(), CommonConstants.SAVE_SUCCESS));
		}
	}

	/**
	 * 插入excel bad record
	 */
	private void insertExcel(TBankSetVo excel) {
		TBankSet insert = new TBankSet();
		BeanUtil.copyProperties(excel, insert);
		this.save(insert);
	}

	@Override
	public List<TBankSet> getList(String bankName, String id) {
		LambdaQueryWrapper<TBankSet> wrapper = Wrappers.lambdaQuery();
		wrapper.eq(TBankSet:: getDeleteFlag, CommonConstants.STATUS_NORMAL);
		wrapper.eq(TBankSet:: getBankName, bankName);
		if (Common.isNotNull(id)) {
			wrapper.ne(TBankSet:: getId, id);
		}
		return baseMapper.selectList(wrapper);
	}

}
